<?php
namespace EdgeCore\CPT\Shortcodes\ImageCarouselItem;

use EdgeCore\Lib;

class ImageCarouselItem implements Lib\ShortcodeInterface {
	private $base;
	
	public function __construct() {
		$this->base = 'edgtf_image_carousel_item';

		add_action('vc_before_init', array($this, 'vcMap'));
	}

	/**
	 * Returns base for shortcode
	 * @return string
	 */
	public function getBase() {
		return $this->base;
	}

	/**
	 * Maps shortcode to Visual Composer. Hooked on vc_before_init
	 */
	public function vcMap() {
		if(function_exists('vc_map')) {
			vc_map(
				array(
					'name'                      => esc_html__( 'Edge Image Carousel Item', 'edgtf-core' ),
					'base'                      => $this->getBase(),
					'category'                  => esc_html__( 'by EDGE', 'edgtf-core' ),
					'icon'                      => 'icon-wpb-image-carousel-item extended-custom-icon',
					'allowed_container_element' => 'vc_row',
					'as_child'                  => array( 'only' => 'edgtf_image_carousel' ),
					'params'                    => array(
						array(
							'type'        => 'attach_image',
							'param_name'  => 'image',
							'heading'     => esc_html__( 'Image', 'edgtf-core' ),
							'description' => esc_html__( 'Select images from media library', 'edgtf-core' )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'title',
							'heading'     => esc_html__( 'Slide Title', 'edgtf-core' ),
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'title_custom',
							'heading'     => esc_html__( 'Use Our Predefined Style For Title', 'edgtf-core' ),
							'value'       => array_flip(gastrobar_edge_get_yes_no_select_array(false, true)),
							'dependency'  => array('element' => 'title', 'not_empty' => true)
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'title_transform',
							'heading'     => esc_html__( 'Slide Title Transform', 'edgtf-core' ),
							'value'       => array_flip(gastrobar_edge_get_text_transform_array(true)),
							'dependency'  => array('element' => 'title', 'not_empty' => true)
						),
						array(
							'type'        => 'colorpicker',
							'param_name'  => 'title_color',
							'heading'     => esc_html__( 'Slide Title Color', 'edgtf-core' ),
							'dependency'  => array('element' => 'title', 'not_empty' => true)
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'custom_link',
							'heading'     => esc_html__( 'Slide Custom Link', 'edgtf-core' ),
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'custom_link_target',
							'heading'    => esc_html__( 'Custom Link Target', 'edgtf-core' ),
							'value'      => array_flip( gastrobar_edge_get_link_target_array() ),
						),
						array(
							'type'        => 'textarea',
							'param_name'  => 'text',
							'heading'     => esc_html__( 'Text', 'edgtf-core' )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'text_offset',
							'heading'     => esc_html__( 'Slides Content Top Offset (px or %)', 'edgtf-core' )
						),
						array(
							'type'        => 'colorpicker',
							'param_name'  => 'text_color',
							'heading'     => esc_html__( 'Slide Text Color', 'edgtf-core' ),
							'dependency'  => array('element' => 'title', 'not_empty' => true)
						),
					)
				)
			);
		}
	}

	/**
	 * Renders shortcodes HTML
	 *
	 * @param $atts array of shortcode params
	 * @param $content string shortcode content
	 * @return string
	 */
	public function render($atts, $content = null) {
		$args = array(
			'image'			          => '',
			'title'			          => '',
			'title_custom'            => 'yes',
			'title_transform'         => '',
			'custom_link'			  => '',
			'custom_link_target'      => '_self',
			'text'                    => '',
			'text_color'              => '',   
			'text_offset'             => '',
		);

		$params = shortcode_atts($args, $atts);

		$params['image_styles']       = $this->getImageStyles($params);
		$params['text_styles']     = $this->getTextStyles($params);
		$params['title_styles']       = $this->getTitleStyles($params);

        $params['custom_link_target'] = !empty($params['custom_link_target']) ? $params['custom_link_target'] : $args['custom_link_target'];

		$html = edgtf_core_get_shortcode_module_template_part('templates/image-carousel-item', 'image-carousel', '', $params);

		return $html;
	}

	/**
	 * Return Image Styles
	 *
	 * @param $params
	 * @return array
	 */
	private function getImageStyles($params) {
		$styles = array();

		$image_src = $this->getImageSrc($params);

		if (!empty($image_src)) {
			$styles[] = 'background-image:  url(' . $image_src.')';
		}

		return implode(';', $styles);
	}

	/**
	 * Return title styles
	 *
	 * @param $params
	 * @return array
	 */
	private function getTitleStyles($params) {
		$styles = array();

		if (!empty($params['title_color'])) {
			$styles[] = 'color:' . $params['title_color'];
		}

		if (!empty($params['title_transform'])) {
			$styles[] = 'text-transform:' . $params['title_transform'];
		}

		return implode(';', $styles);
	}

	/**
	 * Return text styles
	 *
	 * @param $params
	 * @return array
	 */
	private function getTextStyles($params) {
		$styles = array();

		if (!empty($params['text_offset'])) {
			$styles[] = 'margin-top:' . $params['text_offset'];
		}

		if (!empty($params['text_color'])) {
			$styles[] = 'color:' . $params['text_color'];
		}

		return implode(';', $styles);
	}

	/**
	 * Return image url
	 *
	 * @param $params
	 * @return array
	 */
	private function getImageSrc($params) {
		$url = '';

		if ($params['image']) {

			$image['image_id'] = $params['image'];
			$image_original    = wp_get_attachment_image_src($params['image'], 'full');

			$url = $image_original[0];
		}

		return $url;
	}
}