<?php
/*
Plugin Name: HubSpot Blog Integration
Description: Fetch and cache HubSpot blog posts.
Version: 1.0
Author: WP Developer
*/

class HubSpotBlogIntegration {
    private $cache_dir;
    private $log_file;
    private $current_user;
    private $logger;

    public function __construct() {
        $this->cache_dir = plugin_dir_path(__FILE__) . 'cache/';
        // Set up the log file path.
        $this->log_file = plugin_dir_path(__FILE__) . 'logs/hubspot_logs.txt';    
        $this->logger = plugin_dir_path(__FILE__) . 'logger.php';     
        
        // Hook into WordPress to initialize the plugin.
        add_action('init', array($this, 'initialize'));
        add_action('admin_menu', array($this, 'add_settings_page'));
        add_action('rest_api_init', array($this, 'register_webhook_route'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_styles'));
        add_action('wp_enqueue_scripts', array($this, 'front_enqueue_scripts'));
        add_shortcode( 'hubspotBlog', array($this, 'displayBlog') );
    }
    
    public function initialize() {
        $this->current_user = wp_get_current_user();
        // Register settings page for HubSpot API token.
        add_option('hubspot_token', ''); // Initialize option if it doesn't exist.
    }

    public function enqueue_styles($hook) {
        if ($hook === 'toplevel_page_log-viewer') {
            wp_enqueue_style('log-viewer-styles', plugin_dir_url(__FILE__) . 'style.css');
        }
    }

    public function front_enqueue_scripts($hook) {

        if ( ! is_admin() ) {
            // Enqueue your script here
            wp_enqueue_script('hubspot-filter-script', plugins_url('/assets/js/filter.js', __FILE__), array(), '1.0', true);
            $custom_css = get_option('custom_css');
            if (!empty($custom_css)) {
                //die( 'hello' );
                wp_add_inline_style('hubspot-filter-css', $custom_css);
            }
        }
    }
    
    public function add_settings_page() {
        add_menu_page('HS Blog API Config', 'HubSpot Settings', 'manage_options', 'hubspot-settings', array($this, 'display_settings_page'));
    }
    
    public function display_settings_page() {
        if (isset($_POST['hubspot_token'])) {
            // Save the HubSpot authorization token securely.
            $token = sanitize_text_field($_POST['hubspot_token']);
            update_option('hubspot_token', $token);
            echo '<div class="updated"><p>Token saved.</p></div>';
            $this->log( "Token saved by ".$this->current_user->user_email );
        }
        
        $token = get_option('hubspot_token', '');
        
        echo '<div class="wrap">';
        echo '<h1>HubSpot Settings</h1>';
        echo '<form method="post">';
        echo '<h2 for="hubspot_token">HubSpot Authorization Token:</h2>';
        echo '<input type="password" name="hubspot_token" value="' . esc_attr($token) . '" />';
        echo '<input type="submit" class="button-primary" value="Save Token" />';
        echo '</form>';
        $this->handle_cache_refresh();
        $this->additionalCss();
        echo '</div>';        
        $this->display_logs();
    }


    private function handle_cache_refresh() {        

        echo '<div class="wrap">';
        echo '<form method="post">';
        echo '<input type="hidden" name="refresh_cache" value="1" />';
        echo '<input type="submit" class="button-primary" value="Refresh Cache" />';
        if (isset($_POST['refresh_cache'])) {
            // Refresh the cache here.
            $api_data = $this->get_api_data();
            $this->update_cache($api_data);
            echo '<div class="updated"><p>Cache refreshed.</p></div>';
            $this->log( "Cache refreshed by ".$this->current_user->user_email );
        }
        echo '</form>';
        echo '</div>';
    }

    private function additionalCss(){
        if (isset($_POST['custom_css'])) {
            update_option('custom_css', $_POST['custom_css']);
            $this->log( "Css updated by ".$this->current_user->user_email );
        }
        echo '<div class="wrap"><h2>Add Custom CSS</h2><form method="post" action=""><textarea name="custom_css" rows="10" cols="50">'.esc_textarea(get_option("custom_css")).'</textarea><p class="submit"><input type="submit" name="submit" id="submit" class="button button-primary" value="Save Changes"></p></form></div>';
    }

    private function display_logs() {
        if (isset($_POST['clear_logs'])) {
            // Clear the logs if the "Clear Logs" button is clicked.
            $this->clear_logs();
        }

        echo '<div class="wrap">';
        echo '<form method="post">';
        echo '<h2>HubSpot Logs</h2>';
        echo '<input type="hidden" name="clear_logs" value="1" />';
        echo '<p><strong>Logs:</strong></p>';
        //echo '<pre>';
        echo '<code>';
        $logs = $this->get_logs();
        foreach ($logs as $log) {
            echo esc_html($log) . "<br>";
        }
        echo '</code>';
        //echo '</pre>';
        echo '<input type="submit" class="button" value="Clear Logs" />';
        echo '</form>';
        echo '</div>';
    }

    public function log($message) {
        // Log a message to the log file.
        $timestamp = date('Y-m-d H:i:s');
        $log_message = "[$timestamp] $message\n";
        file_put_contents($this->log_file, $log_message, FILE_APPEND);
    }

    private function get_logs() {
        // Retrieve and return logs from the log file.
        if (file_exists($this->log_file) && is_readable($this->log_file)) {
            return explode("\n", file_get_contents($this->log_file));
        }
        return array();
    }

    private function clear_logs() {
        // Clear the log file.
        file_put_contents($this->log_file, '');
        $this->log( "Logs cleared by ".$this->current_user->user_email );
    }
    
    public function fetch_blog_posts() {
        $cached_data = $this->get_cached_data();
        
        if ($cached_data) {
            return $cached_data;
        } else {
            $api_data = $this->get_api_data();
            $this->update_cache($api_data);
            return $api_data;
        }
    }
    
    private function get_cached_data() {
        $cache_file = $this->cache_dir . 'hubspot_cache.json';
        
        if (file_exists($cache_file) && is_readable($cache_file)) {
            $cache_data = file_get_contents($cache_file);
            return json_decode($cache_data, true);
        }
        
        return false;
    }
    
    private function get_api_data() {
        $api_key = get_option('hubspot_token', '');
        
        if (empty($api_key)) {
            // Log the error if the API key is missing.
            $this->log( "HubSpot API key is missing." );
            return array();
        }        

        // Implement API request logic using wp_remote_get().
        $api_url = 'https://api.hubapi.com/cms/v3/blogs/posts?state=PUBLISHED';
        
        $response = wp_remote_get($api_url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
            ),
        ));
        
        if (is_wp_error($response)) {
            // Log API request errors.
            $this->log('HubSpot API request failed: ' . $response->get_error_message());
            return array();
        }
        
        $body = wp_remote_retrieve_body($response);
        $api_data = json_decode($body, true);
        
        return $api_data;
    }
    
    private function update_cache($data) {
        $cache_file = $this->cache_dir . 'hubspot_cache.json';
        file_put_contents($cache_file, json_encode($data));
    }

    public function register_webhook_route() {
        register_rest_route('hubspot-blog/v1', '/receive', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_webhook_request'),
        ));
    }

    public function handle_webhook_request($request) {
        // Handle the incoming webhook data here
        $data = $request->get_json_params();

        // Implement your logic to process the webhook data (e.g., create/update/delete a blog post)
        // Return a response if needed
    }

    public function displayBlog(){
        $blogs = $this->fetch_blog_posts();
        $getTags = $this->getTaglist();
        echo '<section class="elementor-section">';   
        echo '<div class="elementor-container elementor-column-gap-default">
        <div class="elementor-row flex-element-row">
                <div class="elementor-column elementor-col-66">
                    <p class="check-out">I want to check out </p>
                    <div class="elementor-form-field filter-custom-list">
                       '.$getTags.'
                    </div>
                </div>
                <div class="elementor-column elementor-col-6">                    
                    <div class="elementor-field-group">
                        <label for="search-input" class="elementor-screen-only">Search:</label>
                        <input type="text" id="search-input" class="elementor-field elementor-size-sm" name="s" placeholder="Search..." />
                    </div>                   
                </div>
            </div>    
        </div>';
        echo '<div class="elementor-container elementor-column-gap-default elementor-hubspot-blog-rows">';     
        foreach ($blogs['results'] as $key => $value) {
            //echo '<pre>'; print_R($value); echo '<pre>';
            $blogID = $value['id'];
            $tagIds = implode(',', $value['tagIds']);
            $blogTitle = $value['name'];
            $blogImage = $value['featuredImage'];
            $blogAltText = $value['featuredImageAltText'];
            $shortDescription = wpautop(substr(strip_tags($value['postBody']), 0, 100).'...');
            $blogLink = $value['url'];
            if (($key + 1) % 3 === 0 && $key !== count($value) - 1) {
               // echo '</div>';
                 //echo '<div class="elementor-container elementor-column-gap-default">';
            }
            echo '<div class="elementor-column elementor-col-33 elementor-top-column elementor-element elementor-hubspot-post" data-id="'.$blogID.'" data-element_type="column" data-tag-ids="'.$tagIds.'">
                <div class="elementor-widget-wrap elementor-element-populated">
                    <div class="elementor-element elementor-widget elementor-widget-image" data-element_type="widget" data-widget_type="image.default">
                        <div class="elementor-widget-container">
                            <figure class="wp-caption">
                                <a target="_blank" href="'.$blogLink.'"><img decoding="async" src="'.$blogImage.'" title="'.$blogTitle.'" alt="'.$blogAltText.'" loading="lazy">
                                <figcaption class="widget-image-caption wp-caption-text">'.$blogTitle.'</figcaption>
                                </a>
                            </figure>
                            '.$shortDescription.'
                            <a href="'.$blogLink.'" target="_blank" class="elementor-button-link elementor-button elementor-size-sm">
                                <span class="elementor-button-content-wrapper">
                                    <span class="elementor-button-text">Read More</span>
                                </span>
                            </a>
                        </div>
                    </div>
                </div>
            </div>';
        }
        echo '</div></section>';
    }

    private function getTaglist(){
        $tags =   get_terms( array ('taxonomy' => 'post_tag', 'orderby' => 'id','order' => 'ASC','hide_empty' => false ) );
	
        if( isset( $tags ) && !empty( $tags ) ):
            $select = '<select class="elementor-field-textual filter-custom-dropdown" id="filterDropdown" name="dropdown">';
            $select .='<option value="all">Everything</option>';
            foreach ($tags as $value) {
                $select .='<option value="'.$value->slug.'">'.$value->name.'</option>';
            }
            $select .=' </select>';
        else:
            return false;
        endif;
        return $select;
    }

    private function filterScript(){
        
    }
}

// Instantiate the HubSpotBlogIntegration class.
$HB_integrate = new HubSpotBlogIntegration();
