(function($, Drupal, drupalSettings) {

  Drupal.behaviors.openlayers = {
    attach: function(context, settings) {
      var openlayersSettings = drupalSettings.openlayers;
      $(context).find('.openlayers-map').once('ol-map').each(function () {
        var settings = openlayersSettings[this.id];
        var olMapObj = new Drupal.Openlayers(this.id, settings);
        var map = olMapObj.olMap;
      });
    }
  };
  
  Drupal.Openlayers = function(mapid, settings) {
    let self = this;
    this.initialise(mapid, settings);
    return 'xyz';
  };

  Drupal.Openlayers.prototype.initialise = function(mapid, settings) {
    let self = this;
    let styles = self.create_styles(settings.map.styles); 
    let interactions = self.create_interactions(settings.map.interactions);
    let controls = self.create_controls(settings.map.controls);
    let layers = self.create_layers(settings.map.layers, settings.features, styles);
    var extent = layers.extent;
    layers = layers.layers;

    var view = new ol.View({
      center: ol.proj.fromLonLat([0, 0]),
      projection: 'EPSG:3857',
      zoom: 2
    });
    
    //  Create map object, using layers and view defined previously
    self.olMap = new ol.Map({
      interactions: interactions,
      controls: controls,
      target: mapid,
      layers: layers,
      view: view,
      keyboardEventTarget: document             //  Required to get KeyboardPan and KeyboardZoom interactions working.
    });

    //  Check whether the features are editable via a widget
    if (jQuery.isEmptyObject(settings.widget)) {
      //  This section fits the extent of the feature layer within the viewable map.
      self.olMap.getView().fit(extent, {padding: [30, 50, 30, 50]});      
      //  Zoom to formatter setting level if a single marker is being shown
      if (extent[0] == extent [2] && extent[1] == extent[3]) {
        self.olMap.getView().setZoom(settings.zoom);
      }
    } else {
      //  This section deals with the settings for the widget - editing toolbar, etc
      if (settings.widget.fieldType == 'geofield') {
        //  Set visibility and readonly attribute of the input element.
        if (settings.widget.inputShow) {
          $('#' + 'edit-' + settings.widget.valueElement).show();
        } else {
          $('#' + 'edit-' + settings.widget.valueElement).hide();
        }
        $('#' + 'edit-' + settings.widget.valueElement).prop('readonly', !settings.widget.inputEditable);     //  readonly = true, editable = false
        var wkt = $('#' + 'edit-' + settings.widget.valueElement).val();
      }
      if (settings.widget.fieldType == 'geolocation') {
        //  Set visibility and readonly attribute of the input element.
        if (settings.widget.inputShow) {
          $('.' + 'form-item-' + settings.widget.valueElement + '-lat').show();
          $('.' + 'form-item-' + settings.widget.valueElement + '-lng').show();
        } else {
          $('.' + 'form-item-' + settings.widget.valueElement + '-lat').hide();
          $('.' + 'form-item-' + settings.widget.valueElement + '-lng').hide();
        }
        $('#' + 'edit-' + settings.widget.valueElement + '-lat').prop('readonly', !settings.widget.inputEditable);
        $('#' + 'edit-' + settings.widget.valueElement + '-lng').prop('readonly', !settings.widget.inputEditable);

        var lat = $('#' + 'edit-' + settings.widget.valueElement + '-lat').val();
        var lng = $('#' + 'edit-' + settings.widget.valueElement + '-lng').val();
        var wkt = '';
        if (lat != '' && lng != '') {
          var wkt = 'POINT (' + lng + ' ' + lat + ')';
        }
      }
      if (wkt == '') {
        wkt = 'GEOMETRYCOLLECTION EMPTY';
      }     

      var format = new ol.format.WKT({splitCollection: true});
      var features = format.readFeatures(wkt, {
        dataProjection: 'EPSG:4326',
        featureProjection: 'EPSG:3857',
      });

      var source = new ol.source.Vector({
          features: features,
        });
       
      var vector_layer = new ol.layer.Vector({
        source: source,
        style: styles,
      });
      self.olMap.addLayer(vector_layer);
      if (wkt != '' && wkt != 'GEOMETRYCOLLECTION EMPTY') {
        var extent = source.getExtent();
        self.olMap.getView().fit(source.getExtent(), { padding: [30, 30, 30, 30]});
        //  If feature is a single point use the widget settings for initial zoom.
        if (extent[0] == extent [2] && extent[1] == extent[3]) {
          self.olMap.getView().setZoom(settings.widget.center_zoom.zoom);
        } 
      } else {
        //  As the map does not have any features, we set the initial center / zoom as per widget settings
        self.olMap.getView().setCenter(ol.proj.fromLonLat([settings.widget.center_zoom.lon, settings.widget.center_zoom.lat]));
        self.olMap.getView().setZoom(settings.widget.center_zoom.zoom);
      }

      var toolbarSettings = settings.widget.toolbarSettings;
      var editingInteractions = { DrawRegular: false, Transform: false, Split: false, Offset: false};
      $.each(toolbarSettings, function(key, item){
        if (!item) {
          editingInteractions[key] = false;
        }
      });
      
      // Add the editbar
      var edit = new ol.control.EditBar({ source: source, interactions : editingInteractions, edition: true });
      self.olMap.addControl(edit);

      // Add a tooltip
      var tooltip = new ol.Overlay.Tooltip();
      self.olMap.addOverlay(tooltip);

      //  Each time a feature is added, changed, or deleted on the map, the WKT input field need to be updated for the relevtant field / delta.
      source.on('change', function(e){
        var features = source.getFeatures();
        var format = new ol.format.WKT();
        var wktFeatures = format.writeFeatures(source.getFeatures(), {featureProjection: 'EPSG:3857', dataProjection: 'EPSG:4326'});
        if (wktFeatures == 'GEOMETRYCOLLECTION EMPTY') {
          wktFeatures = '';
        }
        if (settings.widget.fieldType == 'geofield') {
          $('#' + 'edit-' + settings.widget.valueElement).val(wktFeatures);
        }
        if (settings.widget.fieldType == 'geolocation') {
          if (wktFeatures == '') {
            $('#' + settings.widget.valueElement + '-lat').val('');
            $('#' + settings.widget.valueElement + '-lng').val('');
          } else {
            wktFeatures = wktFeatures.replace('POINT (', '');
            wktFeatures = wktFeatures.replace('POINT(', '');
            wktFeatures = wktFeatures.replace(')', '');
            var wktPoint = wktFeatures.split(' ');
            $('#' + 'edit-' + settings.widget.valueElement + '-lat').val(wktPoint[1]);
            $('#' + 'edit-' + settings.widget.valueElement + '-lng').val(wktPoint[0]);
          }
        }
      });
    };
    
    //  Layer Switcher
    var layerSwitcher = new ol.control.LayerSwitcher({
      tipLabel: 'Layer Switcher', // Optional label for button
      groupSelectStyle: 'children' // Can be 'children' [default], 'group' or 'none'
    });
    self.olMap.addControl(layerSwitcher);
  };

  Drupal.Openlayers.prototype.create_styles = function(styleInfo) {
    let styles = [];
    $.each(styleInfo, function(key, value){
      if (key == 'Icon') {
        if (value.anchor != undefined && jQuery.type(value.anchor) == 'string') {
          value.anchor = value.anchor.split(',');
        }
        if (value.offset != undefined) {
          value.offset = value.offset.split(',');
        }
        if (value.displacement != undefined) {
          value.displacement = value.displacement.split(',');
        }
      }

      var newStyle = new ol.style[key](value);
      var styleKey = key.toLowerCase();
      if (styleKey == 'icon') {
        styleKey = 'image';
      };
      styles[styleKey] = newStyle;
    });

    return new ol.style.Style(styles);
  }; 

  Drupal.Openlayers.prototype.create_controls = function(controlInfo) {
    var controls = [];
    $.each(controlInfo, function(key, value){
      var newControl = new ol.control[key](value);
      controls.push(newControl);      
    });
    return controls;
  };

  Drupal.Openlayers.prototype.create_interactions = function(interactionInfo) {
    var interactions = [];

    $.each(interactionInfo, function(key, value){
      var newInteraction = new ol.interaction[key](value);
      interactions.push(newInteraction);      
    });
    return interactions;
  }; 

  Drupal.Openlayers.prototype.create_layers = function(layerInfo, features_data, styles) {
    var base_layers = [];
    var overlay_layers = [];

    for (const layerKey in layerInfo) {
      var layer = layerInfo[layerKey];
      var newLayer = null;
      if (layer.layer_type == 'tile') {
        if (layer.source.source_type == 'osm') {
          var newSource = new ol.source.OSM(layer.source.options);
        }
        if (layer.source.source_type == 'stamen') {
          var newSource = new ol.source.Stamen(layer.source.options);
        }
        if (layer.source.source_type == 'xyz') {
          var newSource = new ol.source.XYZ(layer.source.options);
        }

        newLayer = new ol.layer.Tile({
            source: newSource,
            type: layer['type'],
            title: layer['title'],
            visible: layer['visible'],
        });
      }
      base_layers.push(newLayer); 
    }

    //  Now loop through the array of features and create a features layer.
    var features = [];

    var extent = ol.extent.createEmpty();

    $.each(features_data, function(key, value){
      if (value.type == 'wkt') {
        var format = new ol.format.WKT();
        var feature = format.readFeature(value.value, {
          dataProjection: 'EPSG:4326',
          featureProjection: 'EPSG:3857',
        });
        extent = ol.extent.extend(extent, feature.getGeometry().getExtent());
      }
      if (value.type == 'latlon') {
        var geom = new ol.geom.Point(ol.proj.fromLonLat([parseFloat(value.value[1]), parseFloat(value.value[0])], 'EPSG:4326'));
        var feature = new ol.Feature({        
          geometry: geom.transform('EPSG:4326', 'EPSG:3857'),
        });
        extent = ol.extent.extend(extent, feature.getGeometry().getExtent());
      }
      feature.setStyle(styles);
      features.push(feature);
    });

    var layer = new ol.layer.Vector({
      source: new ol.source.Vector({
        features: features
      }),
      title: 'Features',
    });         
    overlay_layers.push(layer);  
    
    var show_layer_groups = true;           //  TODO - link to config screen
    
    if (show_layer_groups) {
      var layers = [
        new ol.layer.Group({
          // A layer must have a title to appear in the layerswitcher
          title: 'Base maps',
          layers: base_layers,
        }),
        new ol.layer.Group({
          // A layer must have a title to appear in the layerswitcher
          title: 'Overlays',
          layers: overlay_layers,
        })
      ];
    } else {
      var layers = base_layers.concat(overlay_layers);
    }
    return {layers:layers, extent:extent};
  };
 
})(jQuery, Drupal, drupalSettings);
