<?php

namespace Drupal\timefield\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin for the 'timefield_mini_calendar_formatter' field formatter.
 *
 * @FieldFormatter(
 *  id = "timefield_mini_calendar_formatter",
 *  label = @Translation("Mini Calendar"),
 *  field_types = {"timefield"}
 * )
 */
class TimeFieldMiniCalendar extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {

    return [
      'display_format' => [
        'separator' => ':',
        'period_separator' => '',
        'period' => 'a',
        'hour' => 'g',
        'minute' => 'i',
      ],
      'column_format' => [
        'separator' => ':',
        'period_separator' => '',
        'period' => 'a',
        'hour' => 'g',
        'minute' => 'i',
      ],
      'first_day' => 'mon',
      'absolute_start' => '8:00 am',
      'absolute_end' => '10:00 pm',
      'range' => 120,
      'adjust_range' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    // @todo Change the autogenerated stub
    $element = parent::settingsForm($form, $form_state);
    $settings = $this->getSettings();

    $element += _timefield_display_format_form('column_format', "Column Time Settings", $settings);

    $element['first_day'] = [
      '#title' => $this->t('First Day of the Week'),
      '#type' => 'select',
      '#options' => _timefield_weekly_summary_days(),
      '#default_value' => $settings['first_day'],
      '#required' => TRUE,
    ];
    $element['absolute_start'] = [
      '#title' => $this->t('Mini Cal Start Time'),
      '#description' => $this->t('The Start Time of the Calendar'),
      '#type' => 'textfield',
      '#default_value' => $settings['absolute_start'],
      '#size' => 15,
      '#maxlength' => 15,
      '#attributes' => [
        'class' => [
          'edit-timefield-timepicker',
          $instance_class,
        ],
      ],
    ];
    $element['absolute_end'] = [
      '#title' => $this->t('Mini Cal End Time'),
      '#description' => $this->t('The End Time of the Calendar'),
      '#type' => 'textfield',
      '#default_value' => $settings['absolute_end'],
      '#size' => 15,
      '#maxlength' => 15,
      '#attributes' => [
        'class' => [
          'edit-timefield-timepicker',
          $instance_class,
        ],
      ],
    ];
    $element['range'] = [
      '#type' => 'select',
      '#title' => $this->t('Select the time duration of each block of time'),
      '#options' => [
        30 => $this->t('30 Minutes'),
        60 => $this->t('1 Hour'),
        90 => $this->t('90 minutes'),
        120 => $this->t('2 Hours'),
        180 => $this->t('3 Hours'),
        240 => $this->t('4 Hours'),
      ],
      '#default_value' => $settings['range'],
    ];
    $element['adjust_range'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Adjust Range to fit Items'),
      '#default_value' => $settings['adjust_time'],
    ];

    $element += _timefield_display_format_form('display_format', "Individual Time Display Settings", $settings);

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    // @todo Change the autogenerated stub
    $summary = parent::settingsSummary();
    $summary[] = $this->t('Current Format: Mini Calendar Format, expand to see current Selection');
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $settings = $this->getSettings();
    $label_hidden = ($this->label == 'hidden');
    $header = [];
    $rows = [];
    foreach ($items as $item) {
      if (!empty($item->value)) {
        $header = _timefield_weekly_summary_build_header($settings['first_day']);
        $rows = timefield_weekly_summary_build_rows($items, $header, $settings);
      }
    }
    return [
      '#theme' => 'timefield_mini_calendar',
      '#label' => $this->fieldDefinition->get('label'),
      '#label_display' => $this->label,
      '#label_hidden' => $label_hidden,
      '#header' => $header,
      '#rows' => $rows,
      '#content' => [
        '#header' => $header,
        '#rows' => $rows,
        '#theme' => 'table',
      ],
    ];
  }

}
