<?php

namespace Civi\Api4\Action\SearchDisplay;

use Civi\API\Exception\UnauthorizedException;
use Civi\Api4\Generic\Traits\ArrayQueryActionTrait;
use Civi\Api4\Query\SqlField;
use Civi\Api4\SearchDisplay;
use Civi\Api4\Utils\CoreUtil;
use Civi\Api4\Utils\FormattingUtil;

/**
 * Base class for running a search.
 *
 * @method $this setDisplay(array|string $display)
 * @method array|string|null getDisplay()
 * @method $this setSort(array $sort)
 * @method array getSort()
 * @method $this setFilters(array $filters)
 * @method array getFilters()
 * @method $this setSeed(string $seed)
 * @method string getSeed()
 * @method $this setAfform(string $afform)
 * @method string getAfform()
 * @package Civi\Api4\Action\SearchDisplay
 */
abstract class AbstractRunAction extends \Civi\Api4\Generic\AbstractAction {

  use SavedSearchInspectorTrait;
  use ArrayQueryActionTrait;

  /**
   * Either the name of the display or an array containing the display definition (for preview mode)
   *
   * Leave NULL to use the autogenerated default.
   *
   * @var string|array|null
   */
  protected $display;

  /**
   * Array of fields to use for ordering the results
   * @var array
   */
  protected $sort = [];

  /**
   * Search conditions that will be automatically added to the WHERE or HAVING clauses
   * @var array
   */
  protected $filters = [];

  /**
   * Filters passed directly into this display via Afform markup
   * will have their labels appended to the Afform title.
   *
   * @var array
   */
  protected $filterLabels = [];

  /**
   * Integer used as a seed when ordering by RAND().
   * This keeps the order stable enough to use a pager with random sorting.
   *
   * @var int
   */
  protected $seed;

  /**
   * Name of Afform, if this display is embedded (used for permissioning)
   * @var string
   */
  protected $afform;

  /**
   * @var array
   */
  private $_afform;

  /**
   * Override execute method to change the result object type
   * @return \Civi\Api4\Result\SearchDisplayRunResult
   */
  public function execute() {
    return parent::execute();
  }

  /**
   * @param \Civi\Api4\Generic\Result $result
   * @throws UnauthorizedException
   * @throws \API_Exception
   */
  public function _run(\Civi\Api4\Generic\Result $result) {
    // Only SearchKit admins can use this in unsecured "preview mode"
    if (
      (is_array($this->savedSearch) || is_array($this->display)) && $this->checkPermissions &&
      !\CRM_Core_Permission::check([['administer CiviCRM data', 'administer search_kit']])
    ) {
      throw new UnauthorizedException('Access denied');
    }
    $this->loadSavedSearch();
    $this->loadSearchDisplay();

    // Displays with acl_bypass must be embedded on an afform which the user has access to
    if (
      $this->checkPermissions && !empty($this->display['acl_bypass']) &&
      !\CRM_Core_Permission::check('all CiviCRM permissions and ACLs') && !$this->loadAfform()
    ) {
      throw new UnauthorizedException('Access denied');
    }

    $this->_apiParams['checkPermissions'] = $this->savedSearch['api_params']['checkPermissions'] = empty($this->display['acl_bypass']);
    $this->display['settings']['columns'] = $this->display['settings']['columns'] ?? [];

    $this->processResult($result);
  }

  abstract protected function processResult(\Civi\Api4\Result\SearchDisplayRunResult $result);

  /**
   * Transforms each row into an array of raw data and an array of formatted columns
   *
   * @param \Civi\Api4\Generic\Result $result
   * @return array{data: array, columns: array}[]
   */
  protected function formatResult(\Civi\Api4\Generic\Result $result): array {
    $rows = [];
    $keyName = CoreUtil::getIdFieldName($this->savedSearch['api_entity']);
    foreach ($result as $index => $record) {
      $data = $columns = [];
      foreach ($this->getSelectClause() as $key => $item) {
        $data[$key] = $this->getValue($key, $record, $index);
      }
      foreach ($this->display['settings']['columns'] as $column) {
        $columns[] = $this->formatColumn($column, $data);
      }
      $style = $this->getCssStyles($this->display['settings']['cssRules'] ?? [], $data);
      $row = [
        'data' => $data,
        'columns' => $columns,
        'cssClass' => implode(' ', $style),
      ];
      if (isset($data[$keyName])) {
        $row['key'] = $data[$keyName];
      }
      $rows[] = $row;
    }
    return $rows;
  }

  /**
   * @param string $key
   * @param array $data
   * @param int $rowIndex
   * @return mixed
   */
  private function getValue($key, $data, $rowIndex) {
    // Get value from api result unless this is a pseudo-field which gets a calculated value
    switch ($key) {
      case 'result_row_num':
        return $rowIndex + 1 + ($this->_apiParams['offset'] ?? 0);

      case 'user_contact_id':
        return \CRM_Core_Session::getLoggedInContactID();

      default:
        if (!empty($data[$key])) {
          $item = $this->getSelectExpression($key);
          if ($item['expr'] instanceof SqlField && $item['fields'][0]['fk_entity'] === 'File') {
            return $this->generateFileUrl($data[$key]);
          }
        }
        return $data[$key] ?? NULL;
    }
  }

  /**
   * Convert file id to a readable url
   *
   * @param $fileID
   * @return string
   * @throws \CRM_Core_Exception
   */
  private function generateFileUrl($fileID) {
    $entityId = \CRM_Core_DAO::getFieldValue('CRM_Core_DAO_EntityFile',
      $fileID,
      'entity_id',
      'file_id'
    );
    $fileHash = \CRM_Core_BAO_File::generateFileHash($entityId, $fileID);
    return $this->getUrl('civicrm/file', [
      'reset' => 1,
      'id' => $fileID,
      'eid' => $entityId,
      'fcs' => $fileHash,
    ]);
  }

  /**
   * @param array $column
   * @param array $data
   * @return array{val: mixed, links: array, edit: array, label: string, title: string, image: array, cssClass: string}
   */
  private function formatColumn($column, $data) {
    $column += ['rewrite' => NULL, 'label' => NULL];
    $out = [];
    switch ($column['type']) {
      case 'field':
        $rawValue = $data[$column['key']] ?? NULL;
        if (!$this->hasValue($rawValue) && isset($column['empty_value'])) {
          $out['val'] = $this->replaceTokens($column['empty_value'], $data, 'view');
        }
        elseif ($column['rewrite']) {
          $out['val'] = $this->rewrite($column, $data);
        }
        else {
          $out['val'] = $this->formatViewValue($column['key'], $rawValue);
        }
        if ($this->hasValue($column['label']) && (!empty($column['forceLabel']) || $this->hasValue($out['val']))) {
          $out['label'] = $this->replaceTokens($column['label'], $data, 'view');
        }
        if (!empty($column['link'])) {
          $links = $this->formatFieldLinks($column, $data, $out['val']);
          if ($links) {
            $out['links'] = $links;
          }
        }
        elseif (!empty($column['editable']) && !$column['rewrite']) {
          $edit = $this->formatEditableColumn($column, $data);
          if ($edit) {
            $out['edit'] = $edit;
          }
        }
        break;

      case 'image':
        $out['img'] = $this->formatImage($column, $data);
        if ($out['img']) {
          $out['val'] = $this->replaceTokens($column['image']['alt'] ?? NULL, $data, 'view');
        }
        if ($this->hasValue($column['label']) && (!empty($column['forceLabel']) || $out['img'])) {
          $out['label'] = $this->replaceTokens($column['label'], $data, 'view');
        }
        if (!empty($column['link'])) {
          $links = $this->formatFieldLinks($column, $data, '');
          if ($links) {
            $out['links'] = $links;
          }
        }
        break;

      case 'links':
      case 'buttons':
      case 'menu':
        $out = $this->formatLinksColumn($column, $data);
        break;
    }
    // Format tooltip
    if (isset($column['title']) && strlen($column['title'])) {
      $out['title'] = $this->replaceTokens($column['title'], $data, 'view');
    }
    $cssClass = $this->getCssStyles($column['cssRules'] ?? [], $data);
    if (!empty($column['alignment'])) {
      $cssClass[] = $column['alignment'];
    }
    if ($cssClass) {
      $out['cssClass'] = implode(' ', $cssClass);
    }
    if (!empty($column['icons'])) {
      $out['icons'] = $this->getColumnIcons($column['icons'], $data);
    }
    return $out;
  }

  /**
   * Rewrite field value, subtituting tokens and evaluating smarty tags
   *
   * @param array $column
   * @param array $data
   * @return string
   */
  private function rewrite(array $column, array $data): string {
    $output = $this->replaceTokens($column['rewrite'], $data, 'view');
    // Cheap strpos to skip Smarty processing if not needed
    if (strpos($output, '{') !== FALSE) {
      $smarty = \CRM_Core_Smarty::singleton();
      $output = $smarty->fetchWith("string:$output", []);
    }
    return $output;
  }

  /**
   * Evaluates conditional style rules
   *
   * Rules are in the format ['css class', 'field_name', 'OPERATOR', 'value']
   *
   * @param array[] $styleRules
   * @param array $data
   * @return array
   */
  protected function getCssStyles(array $styleRules, array $data) {
    $classes = [];
    foreach ($styleRules as $clause) {
      $cssClass = $clause[0] ?? '';
      if ($cssClass) {
        $condition = $this->getRuleCondition(array_slice($clause, 1));
        if (is_null($condition[0]) || (self::filterCompare($data, $condition))) {
          $classes[] = $cssClass;
        }
      }
    }
    return $classes;
  }

  /**
   * Evaluates conditional style rules
   *
   * @param array{icon: string, field: string, if: array, side: string}[] $icons
   * @param array $data
   * @return array
   */
  protected function getColumnIcons(array $icons, array $data) {
    $result = [];
    foreach ($icons as $icon) {
      $iconClass = $icon['icon'] ?? NULL;
      if (!$iconClass && !empty($icon['field'])) {
        $iconClass = $data[$icon['field']] ?? NULL;
      }
      if ($iconClass) {
        $condition = $this->getRuleCondition($icon['if'] ?? []);
        if (!is_null($condition[0]) && !(self::filterCompare($data, $condition))) {
          continue;
        }
        $result[] = ['class' => $iconClass, 'side' => $icon['side'] ?? 'left'];
      }
    }
    return $result;
  }

  /**
   * Returns the condition of a cssRules
   *
   * @param array $clause
   * @return array
   */
  protected function getRuleCondition($clause) {
    $fieldKey = $clause[0] ?? NULL;
    // For fields used in group by, add aggregation and change operator to CONTAINS
    // NOTE: This doesn't support any other operators for aggregated fields.
    if ($fieldKey && $this->canAggregate($fieldKey)) {
      $clause[1] = 'CONTAINS';
      $fieldKey = 'GROUP_CONCAT_' . str_replace(['.', ':'], '_', $clause[0]);
    }
    return [$fieldKey, $clause[1] ?? 'IS NOT EMPTY', $clause[2] ?? NULL];
  }

  /**
   * Return fields needed for the select clause by a set of css rules
   *
   * @param array $cssRules
   * @return array
   */
  protected function getCssRulesSelect($cssRules) {
    $select = [];
    foreach ($cssRules as $clause) {
      $fieldKey = $clause[1] ?? NULL;
      if ($fieldKey) {
        // For fields used in group by, add aggregation
        $select[] = $this->canAggregate($fieldKey) ? "GROUP_CONCAT($fieldKey) AS GROUP_CONCAT_" . str_replace(['.', ':'], '_', $fieldKey) : $fieldKey;
      }
    }
    return $select;
  }

  /**
   * Return fields needed for calculating a column's icons
   *
   * @param array $icons
   * @return array
   */
  protected function getIconsSelect($icons) {
    $select = [];
    foreach ($icons as $icon) {
      if (!empty($icon['field'])) {
        $select[] = $icon['field'];
      }
      $fieldKey = $icon['if'][0] ?? NULL;
      if ($fieldKey) {
        // For fields used in group by, add aggregation
        $select[] = $this->canAggregate($fieldKey) ? "GROUP_CONCAT($fieldKey) AS GROUP_CONCAT_" . str_replace(['.', ':'], '_', $fieldKey) : $fieldKey;
      }
    }
    return $select;
  }

  /**
   * Format a field value as links
   * @param $column
   * @param $data
   * @param $value
   * @return array{text: string, url: string, target: string}[]
   */
  private function formatFieldLinks($column, $data, $value): array {
    $links = [];
    foreach ((array) $value as $index => $val) {
      $path = $this->getLinkPath($column['link'], $data, $index);
      $path = $this->replaceTokens($path, $data, 'url', $index);
      if ($path) {
        $link = [
          'text' => $val,
          'url' => $this->getUrl($path),
        ];
        if (!empty($column['link']['target'])) {
          $link['target'] = $column['link']['target'];
        }
        $links[] = $link;
      }
    }
    return $links;
  }

  /**
   * Format links for a menu/buttons/links column
   * @param array $column
   * @param array $data
   * @return array{text: string, url: string, target: string, style: string, icon: string}[]
   */
  private function formatLinksColumn($column, $data): array {
    $out = ['links' => []];
    if (isset($column['text'])) {
      $out['text'] = $this->replaceTokens($column['text'], $data, 'view');
    }
    foreach ($column['links'] as $item) {
      if (!$this->checkLinkCondition($item, $data)) {
        continue;
      }
      $path = $this->replaceTokens($this->getLinkPath($item, $data), $data, 'url');
      if ($path) {
        $link = [
          'text' => $this->replaceTokens($item['text'] ?? '', $data, 'view'),
          'url' => $this->getUrl($path),
        ];
        foreach (['target', 'style', 'icon'] as $prop) {
          if (!empty($item[$prop])) {
            $link[$prop] = $item[$prop];
          }
        }
        $out['links'][] = $link;
      }
    }
    return $out;
  }

  /**
   * Check if a link should be shown based on its conditions.
   *
   * Given a link, check if it is set to be displayed conditionally.
   * If so, evaluate the condition, else return TRUE.
   *
   * @param array $item
   * @param array $data
   * @return bool
   */
  private function checkLinkCondition(array $item, array $data): bool {
    if (empty($item['condition'][0]) || empty($item['condition'][1])) {
      return TRUE;
    }
    $op = $item['condition'][1];
    if ($item['condition'][0] === 'check user permission') {
      if (!empty($item['condition'][2]) && !\CRM_Core_Permission::check($item['condition'][2])) {
        return $op !== '=';
      }
      return TRUE;
    }
    return self::filterCompare($data, $item['condition']);
  }

  /**
   * @param array $link
   * @param array $data
   * @param int $index
   * @return string|null
   */
  private function getLinkPath($link, $data = NULL, $index = 0) {
    $path = $link['path'] ?? NULL;
    if (!$path && !empty($link['entity']) && !empty($link['action'])) {
      $entity = $link['entity'];
      $idField = $idKey = CoreUtil::getIdFieldName($entity);
      // Hack to support links to relationships
      if ($entity === 'Relationship') {
        $entity = 'RelationshipCache';
        $idKey = 'relationship_id';
      }
      $path = CoreUtil::getInfoItem($entity, 'paths')[$link['action']] ?? NULL;
      $prefix = '';
      if ($path && !empty($link['join'])) {
        $prefix = $link['join'] . '.';
      }
      // This is a bit clunky, the function_join_field gets un-munged later by $this->getJoinFromAlias()
      if ($this->canAggregate($prefix . $idKey)) {
        $prefix = 'GROUP_CONCAT_' . str_replace('.', '_', $prefix);
      }
      if ($prefix) {
        $path = str_replace('[', '[' . $prefix, $path);
      }
      // Check access for edit/update links
      // (presumably if a record is shown in SearchKit the user already has view access, and the check is expensive)
      if ($path && isset($data) && !in_array($link['action'], ['view', 'preview'], TRUE)) {
        $id = $data[$prefix . $idKey] ?? NULL;
        $id = is_array($id) ? $id[$index] ?? NULL : $id;
        if ($id) {
          $access = civicrm_api4($link['entity'], 'checkAccess', [
            // Fudge links with funny action names to check 'update'
            'action' => $link['action'] === 'delete' ? 'delete' : 'update',
            'values' => [
              $idField => $id,
            ],
          ], 0)['access'];
          if (!$access) {
            return NULL;
          }
        }
      }
    }
    return $path;
  }

  /**
   * @param string $path
   * @param array $query
   * @return string
   */
  private function getUrl(string $path, $query = NULL) {
    if ($path[0] === '/' || strpos($path, 'http://') !== FALSE || strpos($path, 'https://') !== FALSE) {
      return $path;
    }
    // Use absolute urls when downloading spreadsheet
    $absolute = $this->getActionName() === 'download';
    return \CRM_Utils_System::url($path, $query, $absolute, NULL, FALSE);
  }

  /**
   * @param array $column
   * @param array $data
   * @return array{entity: string, action: string, input_type: string, data_type: string, options: bool, serialize: bool, nullable: bool, fk_entity: string, value_key: string, record: array, value: mixed}|null
   */
  private function formatEditableColumn($column, $data) {
    $editable = $this->getEditableInfo($column['key']);
    $editable['record'] = [];
    // Generate params to edit existing record
    if (!empty($data[$editable['id_path']])) {
      $editable['action'] = 'update';
      $editable['record'][$editable['id_key']] = $data[$editable['id_path']];
      $editable['value'] = $data[$editable['value_path']];
      // Ensure field is appropriate to this entity sub-type
      $field = $this->getField($column['key']);
      $entityValues = FormattingUtil::filterByPrefix($data, $editable['id_path'], $editable['id_key']);
      if (!$this->fieldBelongsToEntity($editable['entity'], $field['name'], $entityValues)) {
        return NULL;
      }
    }
    // Generate params to create new record, if applicable
    elseif ($editable['explicit_join'] && !$this->getJoin($editable['explicit_join'])['bridge']) {
      $editable['action'] = 'create';
      $editable['value'] = NULL;
      $editable['nullable'] = FALSE;
      // Get values for creation from the join clause
      $join = $this->getQuery()->getExplicitJoin($editable['explicit_join']);
      foreach ($join['on'] ?? [] as $clause) {
        if (is_array($clause) && count($clause) === 3 && $clause[1] === '=') {
          // Because clauses are reversible, check both directions to see which side has a fieldName belonging to this join
          foreach ([0 => 2, 2 => 0] as $field => $value) {
            if (strpos($clause[$field], $editable['explicit_join'] . '.') === 0) {
              $fieldName = substr($clause[$field], strlen($editable['explicit_join']) + 1);
              // If the value is a field, get it from the data
              if (isset($data[$clause[$value]])) {
                $editable['record'][$fieldName] = $data[$clause[$value]];
              }
              // If it's a literal bool or number
              elseif (is_bool($clause[$value]) || is_numeric($clause[$value])) {
                $editable['record'][$fieldName] = $clause[$value];
              }
              // If it's a literal string it will be quoted
              elseif (is_string($clause[$value]) && in_array($clause[$value][0], ['"', "'"], TRUE) && substr($clause[$value], -1) === $clause[$value][0]) {
                $editable['record'][$fieldName] = substr($clause[$value], 1, -1);
              }
            }
          }
        }
      }
      // Ensure all required values exist for create action
      $vals = array_keys(array_filter($editable['record']));
      $vals[] = $editable['value_key'];
      $missingRequiredFields = civicrm_api4($editable['entity'], 'getFields', [
        'action' => 'create',
        'where' => [
          ['type', '=', 'Field'],
          ['required', '=', TRUE],
          ['default_value', 'IS NULL'],
          ['name', 'NOT IN', $vals],
        ],
      ]);
      if ($missingRequiredFields->count() || count($vals) === 1) {
        return NULL;
      }
    }
    // Ensure current user has access
    if ($editable['record']) {
      $access = civicrm_api4($editable['entity'], 'checkAccess', [
        'action' => $editable['action'],
        'values' => $editable['record'],
      ], 0)['access'];
      if ($access) {
        // Remove info that's for internal use only
        \CRM_Utils_Array::remove($editable, 'id_key', 'id_path', 'value_path', 'explicit_join', 'grouping_fields');
        return $editable;
      }
    }
    return NULL;
  }

  /**
   * Check if a field is appropriate for this entity type or sub-type.
   *
   * For example, the 'first_name' field does not belong to Contacts of type Organization.
   * And custom data is sometimes limited to specific contact types, event types, case types, etc.
   *
   * @param string $entityName
   * @param string $fieldName
   * @param array $entityValues
   * @param bool $checkPermissions
   * @return bool
   */
  private function fieldBelongsToEntity($entityName, $fieldName, $entityValues, $checkPermissions = TRUE) {
    try {
      return (bool) civicrm_api4($entityName, 'getFields', [
        'checkPermissions' => $checkPermissions,
        'where' => [['name', '=', $fieldName]],
        'values' => $entityValues,
      ])->count();
    }
    catch (\API_Exception $e) {
      return FALSE;
    }
  }

  /**
   * @param $key
   * @return array{entity: string, input_type: string, data_type: string, options: bool, serialize: bool, nullable: bool, fk_entity: string, value_key: string, value_path: string, id_key: string, id_path: string, explicit_join: string, grouping_fields: array}|null
   */
  private function getEditableInfo($key) {
    $result = NULL;
    // Strip pseudoconstant suffix
    [$key] = explode(':', $key);
    $field = $this->getField($key);
    // If field is an implicit join to another entity (not a custom group), use the original fk field
    if (!empty($field['implicit_join']) && empty($field['custom_field_id'])) {
      return $this->getEditableInfo(substr($key, 0, -1 - strlen($field['name'])));
    }
    if ($field) {
      $idKey = CoreUtil::getIdFieldName($field['entity']);
      $path = ($field['explicit_join'] ? $field['explicit_join'] . '.' : '');
      $idPath = $path . $idKey;
      // Hack to support editing relationships
      if ($field['entity'] === 'RelationshipCache') {
        $field['entity'] = 'Relationship';
        $idPath = $path . 'relationship_id';
      }
      $result = [
        'entity' => $field['entity'],
        'input_type' => $field['input_type'],
        'data_type' => $field['data_type'],
        'options' => !empty($field['options']),
        'serialize' => !empty($field['serialize']),
        'nullable' => !empty($field['nullable']),
        'fk_entity' => $field['fk_entity'],
        'value_key' => $field['name'],
        'value_path' => $key,
        'id_key' => $idKey,
        'id_path' => $idPath,
        'explicit_join' => $field['explicit_join'],
        'grouping_fields' => [],
      ];
      // Grouping fields get added to the query so that contact sub-type and entity type (for custom fields)
      // are available to filter fields specific to an entity sub-type. See self::fieldBelongsToEntity()
      if ($field['type'] === 'Custom' || $field['entity'] === 'Contact') {
        $customInfo = \Civi\Api4\Utils\CoreUtil::getCustomGroupExtends($field['entity']);
        foreach ((array) ($customInfo['grouping'] ?? []) as $grouping) {
          $result['grouping_fields'][] = $path . $grouping;
        }
      }
    }
    return $result;
  }

  /**
   * @param $column
   * @param $data
   * @return array{url: string, width: int, height: int}|NULL
   */
  private function formatImage($column, $data) {
    $tokenExpr = $column['rewrite'] ?: '[' . $column['key'] . ']';
    $url = $this->replaceTokens($tokenExpr, $data, 'url');
    if (!$url && !empty($column['empty_value'])) {
      $url = $this->replaceTokens($column['empty_value'], $data, 'url');
    }
    if (!$url) {
      return NULL;
    }
    return [
      'src' => $url,
      'height' => $column['image']['height'] ?? NULL,
      'width' => $column['image']['width'] ?? NULL,
    ];
  }

  /**
   * @param string $tokenExpr
   * @param array $data
   * @param string $format view|raw|url
   * @param int $index
   * @return string
   */
  private function replaceTokens($tokenExpr, $data, $format, $index = 0) {
    if (strpos($tokenExpr, '[') !== FALSE) {
      foreach ($this->getTokens($tokenExpr) as $token) {
        $val = $data[$token] ?? NULL;
        if (isset($val) && $format === 'view') {
          $val = $this->formatViewValue($token, $val);
        }
        $replacement = is_array($val) ? $val[$index] ?? '' : $val;
        // A missing token value in a url invalidates it
        if ($format === 'url' && (!isset($replacement) || $replacement === '')) {
          return NULL;
        }
        $tokenExpr = str_replace('[' . $token . ']', $replacement, $tokenExpr);
      }
    }
    return $tokenExpr;
  }

  /**
   * Format raw field value according to data type
   * @param string $key
   * @param mixed $rawValue
   * @return array|string
   */
  protected function formatViewValue($key, $rawValue) {
    if (is_array($rawValue)) {
      return array_map(function($val) use ($key) {
        return $this->formatViewValue($key, $val);
      }, $rawValue);
    }

    $dataType = $this->getSelectExpression($key)['dataType'] ?? NULL;

    $formatted = $rawValue;

    switch ($dataType) {
      case 'Boolean':
        if (is_bool($rawValue)) {
          $formatted = $rawValue ? ts('Yes') : ts('No');
        }
        break;

      case 'Money':
        $formatted = \CRM_Utils_Money::format($rawValue);
        break;

      case 'Date':
      case 'Timestamp':
        $formatted = \CRM_Utils_Date::customFormat($rawValue);
    }

    return $formatted;
  }

  /**
   * Applies supplied filters to the where clause
   */
  protected function applyFilters() {
    // Allow all filters that are included in SELECT clause or are fields on the Afform.
    $fieldFilters = $this->getAfformFilterFields();
    $directiveFilters = $this->getAfformDirectiveFilters();
    $allowedFilters = array_merge($this->getSelectAliases(), $fieldFilters, $directiveFilters);

    // Ignore empty strings
    $filters = array_filter($this->filters, [$this, 'hasValue']);
    if (!$filters) {
      return;
    }

    foreach ($filters as $key => $value) {
      $fieldNames = explode(',', $key);
      if (in_array($key, $allowedFilters, TRUE) || !array_diff($fieldNames, $allowedFilters)) {
        $this->applyFilter($fieldNames, $value);
      }
      // Filter labels are used to set the page title for drilldown forms
      if (in_array($key, $directiveFilters, TRUE)) {
        $this->addFilterLabel($key, $value);
      }
    }
  }

  /**
   * Returns an array of field names or aliases + allowed suffixes from the SELECT clause
   * @return string[]
   */
  protected function getSelectAliases() {
    $result = [];
    $selectAliases = array_map(function($select) {
      return array_slice(explode(' AS ', $select), -1)[0];
    }, $this->savedSearch['api_params']['select']);
    foreach ($selectAliases as $alias) {
      [$alias] = explode(':', $alias);
      $result[] = $alias;
      foreach (['name', 'label', 'abbr'] as $allowedSuffix) {
        $result[] = $alias . ':' . $allowedSuffix;
      }
    }
    return $result;
  }

  /**
   * @param array $fieldNames
   *   If multiple field names are given they will be combined in an OR clause
   * @param mixed $value
   */
  private function applyFilter(array $fieldNames, $value) {
    // Global setting determines if % wildcard should be added to both sides (default) or only the end of a search string
    $prefixWithWildcard = \Civi::settings()->get('includeWildCardInName');

    // Based on the first field, decide which clause to add this condition to
    $fieldName = $fieldNames[0];
    $field = $this->getField($fieldName);
    // If field is not found it must be an aggregated column & belongs in the HAVING clause.
    if (!$field) {
      $this->_apiParams += ['having' => []];
      $clause =& $this->_apiParams['having'];
    }
    // If field belongs to an EXCLUDE join, it should be added as a join condition
    else {
      $prefix = strpos($fieldName, '.') ? explode('.', $fieldName)[0] : NULL;
      foreach ($this->_apiParams['join'] ?? [] as $idx => $join) {
        if (($join[1] ?? 'LEFT') === 'EXCLUDE' && (explode(' AS ', $join[0])[1] ?? '') === $prefix) {
          $clause =& $this->_apiParams['join'][$idx];
        }
      }
    }
    // Default: add filter to WHERE clause
    if (!isset($clause)) {
      $clause =& $this->_apiParams['where'];
    }

    $filterClauses = [];

    foreach ($fieldNames as $fieldName) {
      $field = $this->getField($fieldName);
      $dataType = $field['data_type'] ?? NULL;
      // Array is either associative `OP => VAL` or sequential `IN (...)`
      if (is_array($value)) {
        $value = array_filter($value, [$this, 'hasValue']);
        // If array does not contain operators as keys, assume array of values
        if (array_diff_key($value, array_flip(CoreUtil::getOperators()))) {
          // Use IN for regular fields
          if (empty($field['serialize'])) {
            $filterClauses[] = [$fieldName, 'IN', $value];
          }
          // Use an OR group of CONTAINS for array fields
          else {
            $orGroup = [];
            foreach ($value as $val) {
              $orGroup[] = [$fieldName, 'CONTAINS', $val];
            }
            $filterClauses[] = ['OR', $orGroup];
          }
        }
        // Operator => Value array
        else {
          $andGroup = [];
          foreach ($value as $operator => $val) {
            $andGroup[] = [$fieldName, $operator, $val];
          }
          $filterClauses[] = ['AND', $andGroup];
        }
      }
      elseif (!empty($field['serialize'])) {
        $filterClauses[] = [$fieldName, 'CONTAINS', $value];
      }
      elseif (!empty($field['options']) || in_array($dataType, ['Integer', 'Boolean', 'Date', 'Timestamp'])) {
        $filterClauses[] = [$fieldName, '=', $value];
      }
      elseif ($prefixWithWildcard) {
        $filterClauses[] = [$fieldName, 'CONTAINS', $value];
      }
      else {
        $filterClauses[] = [$fieldName, 'LIKE', $value . '%'];
      }
    }
    // Single field
    if (count($filterClauses) === 1) {
      $clause[] = $filterClauses[0];
    }
    else {
      $clause[] = ['OR', $filterClauses];
    }
  }

  /**
   * Transforms the SORT param (which is expected to be an array of arrays)
   * to the ORDER BY clause (which is an associative array of [field => DIR]
   *
   * @return array
   */
  protected function getOrderByFromSort() {
    // Drag-sortable tables have a forced order
    if (!empty($this->display['settings']['draggable'])) {
      return [$this->display['settings']['draggable'] => 'ASC'];
    }

    $defaultSort = $this->display['settings']['sort'] ?? [];
    $currentSort = [];

    // Add requested sort after verifying it corresponds to sortable columns
    foreach ($this->sort as $item) {
      $column = array_column($this->display['settings']['columns'], NULL, 'key')[$item[0]] ?? NULL;
      if ($column && !(isset($column['sortable']) && !$column['sortable'])) {
        $currentSort[] = $item;
      }
    }

    $orderBy = [];
    foreach ($currentSort ?: $defaultSort as $item) {
      // Apply seed to random sorting
      if ($item[0] === 'RAND()' && isset($this->seed)) {
        $item[0] = 'RAND(' . $this->seed . ')';
      }
      // Prevent errors trying to orderBy nonaggregated columns when using groupBy
      if ($this->canAggregate($item[0])) {
        continue;
      }
      $orderBy[$item[0]] = $item[1];
    }
    return $orderBy;
  }

  /**
   * Adds additional fields to the select clause required to render the display
   *
   * @param array $apiParams
   */
  protected function augmentSelectClause(&$apiParams): void {
    $existing = array_map(function($item) {
      return explode(' AS ', $item)[1] ?? $item;
    }, $apiParams['select']);
    $additions = [];
    // Add primary key field if actions are enabled
    // (only needed for non-dao entities, as Api4SelectQuery will auto-add the id)
    if (!in_array('DAOEntity', CoreUtil::getInfoItem($this->savedSearch['api_entity'], 'type')) &&
      (!empty($this->display['settings']['actions']) || !empty($this->display['settings']['draggable']))
    ) {
      $additions = CoreUtil::getInfoItem($this->savedSearch['api_entity'], 'primary_key');
    }
    // Add draggable column (typically "weight")
    if (!empty($this->display['settings']['draggable'])) {
      $additions[] = $this->display['settings']['draggable'];
    }
    // Add style conditions for the display
    foreach ($this->getCssRulesSelect($this->display['settings']['cssRules'] ?? []) as $addition) {
      $additions[] = $addition;
    }
    $possibleTokens = '';
    foreach ($this->display['settings']['columns'] as $column) {
      // Collect display values in which a token is allowed
      $possibleTokens .= ($column['rewrite'] ?? '');
      $possibleTokens .= ($column['title'] ?? '');
      $possibleTokens .= ($column['empty_value'] ?? '');
      if (!empty($column['link'])) {
        $possibleTokens .= $this->getLinkPath($column['link']) ?? '';
      }
      foreach ($column['links'] ?? [] as $link) {
        $possibleTokens .= $link['text'] ?? '';
        $possibleTokens .= $this->getLinkPath($link) ?? '';
      }

      // Select id & value for in-place editing
      if (!empty($column['editable'])) {
        $editable = $this->getEditableInfo($column['key']);
        if ($editable) {
          $additions = array_merge($additions, $editable['grouping_fields'], [$editable['value_path'], $editable['id_path']]);
        }
      }
      // Add style & icon conditions for the column
      $additions = array_merge($additions,
        $this->getCssRulesSelect($column['cssRules'] ?? []),
        $this->getIconsSelect($column['icons'] ?? [])
      );
    }
    // Add fields referenced via token
    $tokens = $this->getTokens($possibleTokens);
    // Only add fields not already in SELECT clause
    $additions = array_diff(array_merge($additions, $tokens), $existing);
    // Tokens for aggregated columns start with 'GROUP_CONCAT_'
    foreach ($additions as $index => $alias) {
      if (strpos($alias, 'GROUP_CONCAT_') === 0) {
        $additions[$index] = 'GROUP_CONCAT(' . $this->getJoinFromAlias(explode('_', $alias, 3)[2]) . ') AS ' . $alias;
      }
    }
    $this->_selectClause = NULL;
    $apiParams['select'] = array_unique(array_merge($apiParams['select'], $additions));
  }

  /**
   * @param string $str
   */
  private function getTokens($str) {
    $tokens = [];
    preg_match_all('/\\[([^]]+)\\]/', $str, $tokens);
    return array_unique($tokens[1]);
  }

  /**
   * Given an alias like Contact_Email_01_location_type_id
   * this will return Contact_Email_01.location_type_id
   * @param string $alias
   * @return string
   */
  protected function getJoinFromAlias(string $alias) {
    $result = '';
    foreach ($this->_apiParams['join'] ?? [] as $join) {
      $joinName = explode(' AS ', $join[0])[1];
      if (strpos($alias, $joinName) === 0) {
        $parsed = $joinName . '.' . substr($alias, strlen($joinName) + 1);
        // Ensure we are using the longest match
        if (strlen($parsed) > strlen($result)) {
          $result = $parsed;
        }
      }
    }
    return $result ?: $alias;
  }

  /**
   * Checks if a filter contains a non-empty value
   *
   * "Empty" search values are [], '', and NULL.
   * Also recursively checks arrays to ensure they contain at least one non-empty value.
   *
   * @param $value
   * @return bool
   */
  private function hasValue($value) {
    return $value !== '' && $value !== NULL && (!is_array($value) || array_filter($value, [$this, 'hasValue']));
  }

  /**
   * Returns a list of afform fields used as search filters
   *
   * Limited to the current display
   *
   * @return string[]
   */
  private function getAfformFilterFields() {
    $afform = $this->loadAfform();
    if ($afform) {
      return array_column(\CRM_Utils_Array::findAll(
        $afform['searchDisplay']['fieldset'],
        ['#tag' => 'af-field']
      ), 'name');
    }
    return [];
  }

  /**
   * Finds all directive filters and applies the ones with a literal value
   *
   * Returns the list of filters that did not get auto-applied (value was passed via js)
   *
   * @return string[]
   */
  private function getAfformDirectiveFilters() {
    $afform = $this->loadAfform();
    if (!$afform) {
      return [];
    }
    $filterKeys = [];
    // Get filters passed into search display directive from Afform markup
    $filterAttr = $afform['searchDisplay']['filters'] ?? NULL;
    if ($filterAttr && is_string($filterAttr) && $filterAttr[0] === '{') {
      foreach (\CRM_Utils_JS::decode($filterAttr) as $filterKey => $filterVal) {
        // Automatically apply filters from the markup if they have a value
        if ($filterVal !== NULL) {
          unset($this->filters[$filterKey]);
          if ($this->hasValue($filterVal)) {
            $this->applyFilter(explode(',', $filterKey), $filterVal);
          }
        }
        // If it's a javascript variable it will have come back from decode() as NULL;
        // whitelist it to allow it to be passed to this api from javascript.
        else {
          $filterKeys[] = $filterKey;
        }
      }
    }
    return $filterKeys;
  }

  /**
   * Return afform with name specified in api call.
   *
   * Verifies the searchDisplay is embedded in the afform and the user has permission to view it.
   *
   * @return array|false
   */
  private function loadAfform() {
    // Only attempt to load afform once.
    if ($this->afform && !isset($this->_afform)) {
      $this->_afform = FALSE;
      // Permission checks are enabled in this api call to ensure the user has permission to view the form
      $afform = \Civi\Api4\Afform::get($this->getCheckPermissions())
        ->addWhere('name', '=', $this->afform)
        ->setLayoutFormat('deep')
        ->execute()->first();
      if (empty($afform['layout'])) {
        return FALSE;
      }
      // Get all search display fieldsets (which will have an empty value for the af-fieldset attribute)
      $fieldsets = \CRM_Utils_Array::findAll($afform['layout'], ['af-fieldset' => '']);
      // As a fallback, search the entire afform in case the search display is not in a fieldset
      $fieldsets['form'] = $afform['layout'];
      // Validate that the afform contains this search display
      foreach ($fieldsets as $key => $fieldset) {
        $afform['searchDisplay'] = \CRM_Utils_Array::findAll(
            $fieldset,
            ['#tag' => $this->display['type:name'], 'search-name' => $this->savedSearch['name'], 'display-name' => $this->display['name']]
          )[0] ?? NULL;
        if ($afform['searchDisplay']) {
          // Set the fieldset for this display (if it is in one and we haven't fallen back to the whole form)
          $afform['searchDisplay']['fieldset'] = $key === 'form' ? [] : $fieldset;
          return $this->_afform = $afform;
        }
      }
    }
    return $this->_afform;
  }

  /**
   * Extra calculated fields provided by SearchKit
   * @return array[]
   */
  public static function getPseudoFields(): array {
    return [
      [
        'name' => 'result_row_num',
        'fieldName' => 'result_row_num',
        'title' => ts('Row Number'),
        'label' => ts('Row Number'),
        'description' => ts('Index of each row, starting from 1 on the first page'),
        'type' => 'Pseudo',
        'data_type' => 'Integer',
        'readonly' => TRUE,
      ],
      [
        'name' => 'user_contact_id',
        'fieldName' => 'user_contact_id',
        'title' => ts('Current User ID'),
        'label' => ts('Current User ID'),
        'description' => ts('Contact ID of the current user if logged in'),
        'type' => 'Pseudo',
        'data_type' => 'Integer',
        'readonly' => TRUE,
      ],
      [
        'name' => 'CURDATE()',
        'fieldName' => 'CURDATE()',
        'title' => ts('Current Date'),
        'label' => ts('Current Date'),
        'description' => ts('System date at the moment the search is run'),
        'type' => 'Pseudo',
        'data_type' => 'Date',
        'readonly' => TRUE,
      ],
      [
        'name' => 'NOW()',
        'fieldName' => 'NOW()',
        'title' => ts('Current Date + Time'),
        'label' => ts('Current Date + Time'),
        'description' => ts('System date and time at the moment the search is run'),
        'type' => 'Pseudo',
        'data_type' => 'Timestamp',
        'readonly' => TRUE,
      ],
    ];
  }

  /**
   * Sets $this->filterLabels to provide contextual titles to search Afforms
   *
   * @param $fieldName
   * @param $value
   * @throws \API_Exception
   * @throws \Civi\API\Exception\NotImplementedException
   */
  private function addFilterLabel($fieldName, $value) {
    $field = $this->getField($fieldName);
    if (!$field || !$value) {
      return;
    }
    $idField = CoreUtil::getIdFieldName($field['entity']);
    if ($field['name'] === $idField) {
      $field['fk_entity'] = $field['entity'];
    }
    if (!empty($field['options'])) {
      $options = civicrm_api4($field['entity'], 'getFields', [
        'loadOptions' => TRUE,
        'checkPermissions' => FALSE,
        'where' => [['name', '=', $field['name']]],
      ])->first()['options'] ?? [];
      foreach ((array) $value as $val) {
        if (!empty($options[$val])) {
          $this->filterLabels[] = $options[$val];
        }
      }
    }
    elseif (!empty($field['fk_entity'])) {
      $idField = CoreUtil::getIdFieldName($field['fk_entity']);
      $labelField = CoreUtil::getInfoItem($field['fk_entity'], 'label_field');
      if ($labelField) {
        $records = civicrm_api4($field['fk_entity'], 'get', [
          'checkPermissions' => $this->checkPermissions,
          'where' => [[$idField, 'IN', (array) $value]],
          'select' => [$labelField],
        ]);
        foreach ($records as $record) {
          if (isset($record[$labelField])) {
            $this->filterLabels[] = $record[$labelField];
          }
        }
      }
    }
  }

  /**
   * Loads display if not already an array
   */
  private function loadSearchDisplay(): void {
    // Display name given
    if (is_string($this->display)) {
      $this->display = SearchDisplay::get(FALSE)
        ->setSelect(['*', 'type:name'])
        ->addWhere('name', '=', $this->display)
        ->addWhere('saved_search_id', '=', $this->savedSearch['id'])
        ->execute()->single();
    }
    // Null given - use default display
    elseif (is_null($this->display)) {
      $this->display = SearchDisplay::getDefault(FALSE)
        ->addSelect('*', 'type:name')
        ->setSavedSearch($this->savedSearch)
        ->execute()->first();
    }
  }

}
