<?php

namespace Drupal\openlayers\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\openlayers\ConfigurableImageEffectInterface;
use Drupal\openlayers\OpenlayersMapInterface;
use Drupal\openlayers\OpenlayersLayerInterface;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

use Drupal\openlayers\MapInteraction;

/**
 * Provides a base form for image effects.
 */
abstract class MapInteractionFormBase extends FormBase {

  /**
   * The image style.
   *
   * @var \Drupal\image\ImageStyleInterface
   */
  protected $olMap;

  /**
   * The image effect.
   *
   * @var \Drupal\image\ImageEffectInterface|\Drupal\image\ConfigurableImageEffectInterface
   */
  protected $olLayer;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ol_map_layer_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\image\ImageStyleInterface $image_style
   *   The image style.
   * @param string $image_effect
   *   The image effect ID.
   *
   * @return array
   *   The form structure.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   */
  public function buildForm(array $form, FormStateInterface $form_state, OpenlayersMapInterface $map = NULL, $layer = NULL) {
    $request = $this->getRequest();
    $user_input = $form_state->getUserInput();
    $form['#attached']['library'][] = 'openlayers/admin';
    
    $this->olMap = $map;

    if (isset($this->olMap->layers[$layer])) {
      $this->mapLayer = new MapLayer($this->olMap->layers[$layer], 'update');
    } else {
      $this->mapLayer = new MapLayer($layer, 'add');
    }

    $form['uuid'] = [
      '#type' => 'value',
      '#value' => $this->mapLayer->getUuid(),
    ];
    $form['id'] = [
      '#type' => 'value',
      '#value' => $this->mapLayer->id(),
    ];

    $form['data'] = [];
    $subform_state = SubformState::createForSubform($form['data'], $form, $form_state);
    $form['data'] = $this->mapLayer->buildConfigurationForm($form['data'], $subform_state); 
    $form['data']['#tree'] = TRUE;

    // Check the URL for a weight, then the image effect, otherwise use default.
    $form['weight'] = [
      '#type' => 'hidden',
      '#value' => 0,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#button_type' => 'primary',
    ];
    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => $this->olMap->toUrl('edit-form'),
      '#attributes' => ['class' => ['button']],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // The image effect configuration is stored in the 'data' key in the form,
    // pass that through for validation.
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {    
    $form_state->cleanValues();

    // The image effect configuration is stored in the 'data' key in the form,
    // pass that through for submission.
    $this->mapLayer->submitConfigurationForm($form['data'], SubformState::createForSubform($form['data'], $form, $form_state));
    $this->mapLayer->setWeight($form_state->getValue('weight'));
    if (!$this->mapLayer->getUuid()) {
      $this->olMap->addMapLayer($this->mapLayer->getConfiguration());     //  TODO - where is the addMapLayer function ?
    } else {
      $this->olMap->updateMapLayer($this->mapLayer->getConfiguration());
    }
    $this->olMap->save();
    $this->messenger()->addStatus($this->t('The image effect was successfully applied.'));
    $form_state->setRedirectUrl($this->olMap->toUrl('edit-form'));
  }

  /**
   * Converts an image effect ID into an object.
   *
   * @param string $image_effect
   *   The image effect ID.
   *
   * @return \Drupal\image\ImageEffectInterface
   *   The image effect object.
   */
  abstract protected function prepareOpenlayersLayer($layer);

}
