<?php

namespace Drupal\openlayers\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

use Drupal\Core\Link;
use Drupal\Component\Utility\Unicode;
use Drupal\Core\Url;

use Drupal\openlayers\StylePluginManager;
use Drupal\openlayers\ControlPluginManager;
use Drupal\openlayers\InteractionPluginManager;

use Drupal\openlayers\OpenlayersPluginCollection;
use Drupal\openlayers\OpenlayersConfigurablePluginInterface;

/**
 * Base form for Openlayers Maps add and edit configuration forms.
 */
class OpenlayersMapFormBase extends EntityForm {
  
  /**
   * The entity being used by this form.
   *
   * @var \Drupal\image\ImageStyleInterface
   */
  protected $entity;

  /**
   * The image style entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
//  protected $imageStyleStorage;
  protected $entityStorage;

  /**
   * The image effect manager service.
   *
   * @var \Drupal\image\ImageEffectManager
   */
  protected $stylePluginManager;
  protected $controlPluginManager;
  protected $interactionPluginManager;
  
  /**
   * Constructs a base class for image style add and edit forms.
   *
   * @param \Drupal\Core\Entity\EntityStorageInterface $image_style_storage
   *   The image style entity storage.
   */
  public function __construct(
      EntityStorageInterface $entity_storage, 
      StylePluginManager $style_plugin_manager, 
      ControlPluginManager $control_plugin_manager, 
      InteractionPluginManager $interaction_plugin_manager
    ) {
    $this->entityStorage = $entity_storage;
    $this->stylePluginManager = $style_plugin_manager;
    $this->controlPluginManager = $control_plugin_manager;
    $this->interactionPluginManager = $interaction_plugin_manager;
  }
  
  public static function create(ContainerInterface $container) {
    $form = new static(
      $container->get('entity_type.manager')->getStorage('openlayers_map'),
      $container->get('plugin.manager.openlayers.style'),
      $container->get('plugin.manager.openlayers.control'),
      $container->get('plugin.manager.openlayers.interaction')
    );
    $form->setMessenger($container->get('messenger'));
    return $form;
  }

  public function form(array $form, FormStateInterface $form_state) {
    $user_input = $form_state->getUserInput();
    $form['#title'] = $this->t('Edit %name map', ['%name' => $this->entity->label()]);
    $form['#tree'] = TRUE;
    $form['#attached']['library'][] = 'openlayers/admin';

    $openlayers_map = $this->entity;

    // Build the form.
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $openlayers_map->label(),
      '#required' => TRUE,
    ];
    $form['id'] = [
      '#type' => 'machine_name',
      '#title' => $this->t('Machine name'),
      '#default_value' => $openlayers_map->id(),
      '#machine_name' => [
        'exists' => [$this, 'exists'],
        'replace_pattern' => '([^a-z0-9_]+)|(^custom$)',
        'error' => 'The machine-readable name must be unique, and can only contain lowercase letters, numbers, and underscores. Additionally, it can not be the reserved word "custom".',
      ],
      '#disabled' => !$openlayers_map->isNew(),
    ];

    //  Start of layers subform
    // Build the list of existing layers for this map.
    $form['layers'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Layers'),
        $this->t('Weight'),
        $this->t('Operations'),
      ],
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'ol-layer-order-weight',
        ],
      ],
      '#attributes' => [
        'id' => 'ol-map-layers',
      ],
      '#empty' => t('There are currently no layers in this map. Add one by selecting an option below.'),
      '#weight' => 5,
    ];

    $all_layers = $this->entity->getAllLayers();
    $map_layers = $this->entity->getLayers();

    uasort($map_layers, function ($a, $b) {
      return (isset($a['weight']) ? $a['weight'] : 10) - (isset($b['weight']) ? $b['weight'] : 10);
    });


    foreach ($map_layers as $key => $layer_info) {

      if (isset($all_layers[$layer_info['id']])) {
        unset($all_layers[$layer_info['id']]);  
      }

      $form['layers'][$key]['#attributes']['class'][] = 'draggable';

      $form['layers'][$key]['layers'] = [
        '#tree' => FALSE,
        'data' => [
          'label' => [
            '#plain_text' => \Drupal::config('openlayers.layer.' . $layer_info['id'])->get('label'),
          ],
        ],
      ];

      $form['layers'][$key]['weight'] = [
        '#type' => 'weight',

        '#title' => $this->t('Weight for @title', ['@title' => 12]),
        '#title_display' => 'invisible',

        '#default_value' => isset($layer_info['weight']) ? $layer_info['weight'] : 10,
        '#attributes' => [
          'class' => ['ol-layer-order-weight'],
        ],
      ];


      $links = [];

      $is_configurable = TRUE;        //  TEMP LINE
      if ($is_configurable) {
        $links['edit'] = [
          'title' => $this->t('Edit'),          
          'url' => Url::fromRoute('openlayers.map.layer_edit_form', [
            'map' => $this->entity->id(),
            'layer' => $key,
          ]),
        ];
      }
      $links['delete'] = [
        'title' => $this->t('Delete'),
        'url' => Url::fromRoute('openlayers.map.layer_delete', [
            'map' => $this->entity->id(),
            'layer' => $key,
        ]),
      ];
      $form['layers'][$key]['operations'] = [
        '#type' => 'operations',
        '#links' => $links,
      ];
    }

    // Build the new layer addition form and add it to the layer list.
    asort($all_layers);

    $form['layers']['new'] = [
      '#tree' => FALSE,
      '#weight' => isset($user_input['weight']) ? $user_input['weight'] : NULL,       //  TODO - ???
      '#attributes' => ['class' => ['draggable']],
    ];
    $form['layers']['new']['layer'] = [
      'data' => [
        'new_layer' => [
          '#type' => 'select',
          '#title' => $this->t('Layer'),
          '#title_display' => 'invisible',
          '#options' => $all_layers,
          '#empty_option' => $this->t('Select a new layer'),
        ],
        [
          'add' => [
            '#type' => 'submit',
            '#value' => $this->t('Add layer'),
            '#name' => 'add_layer',
            '#validate' => ['::layerValidate'],
            '#submit' => ['::submitForm', '::layerSave'],
          ],
        ],
      ],
      '#prefix' => '<div class="map-layer-new">',
      '#suffix' => '</div>',
    ];

    $form['layers']['new']['weight'] = [
      '#type' => 'weight',
      '#title' => $this->t('Weight for new layer'),
      '#title_display' => 'invisible',
      '#default_value' => count($this->entity->getLayers()),
      '#attributes' => ['class' => ['ol-layer-order-weight']],
    ];
    $form['layers']['new']['operations'] = [
      'data' => [],
    ];
    //  End of layers subform

    //  Start of styles subform
    // Build the list of existing styles for this map.
    $form['styles'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Styles'),
        $this->t('Operations'),
      ],
      '#attributes' => [
        'id' => 'ol-map-styles',
      ],
      '#empty' => t('There are currently no styles defined in this map. Add one by selecting an option below.'),
      '#weight' => 5,
    ];


    // Build the list of new styles that can be added to the map, excluding those that are already attached to the map.
    $new_style_options = [];
    $styles = $this->stylePluginManager->getDefinitions();

    uasort($styles, function ($a, $b) {
      return Unicode::strcasecmp($a['label'], $b['label']);
    });
    foreach ($styles as $style => $definition) {
      $new_style_options[$style] = $definition['label'];
    }
 

    $map_styles = $this->entity->getStyles();

    foreach ($map_styles as $key => $style) {

      if (isset($new_style_options[$style->getPluginId()])) {
        unset($new_style_options[$style->getPluginId()]);  
      }
      
      $form['styles'][$key]['style'] = [
        '#tree' => FALSE,
        'data' => [
          'label' => [
            '#plain_text' => $style->label(),
          ],
        ],
      ];

      $links = [];
      $is_configurable = $style instanceof OpenlayersConfigurablePluginInterface;

      if ($is_configurable) {
        $links['edit'] = [
          'title' => $this->t('Edit'),
          'url' => Url::fromRoute('openlayers.plugin_edit_form', [
            'map' => $this->entity->id(),
            'plugin_type' => 'style',
            'plugin' => $key,
          ]),
        ];
      }
      $links['delete'] = [
        'title' => $this->t('Delete'),
        'url' => Url::fromRoute('openlayers.map.style_delete', [
            'map' => $this->entity->id(),
            'style' => $key,
        ]),
      ];
      $form['styles'][$key]['operations'] = [
        '#type' => 'operations',
        '#links' => $links,
      ];
    }

    // Build the new style addition form and add it to the style list.
    $form['styles']['new'] = [
      '#tree' => FALSE,
    ];
    
    $form['styles']['new']['style'] = [
      'data' => [
        'new_style' => [
          '#type' => 'select',
          '#title' => $this->t('Style'),
          '#title_display' => 'invisible',
          '#options' => $new_style_options,
          '#empty_option' => $this->t('Select a new style'),
        ],
        [
          'add' => [
            '#type' => 'submit',
            '#value' => $this->t('Add style'),
            '#name' => 'add_style',
            '#validate' => ['::styleValidate'],
            '#submit' => ['::submitForm', '::styleSave'],
          ],
        ],
      ],
      '#prefix' => '<div class="map-style-new">',
      '#suffix' => '</div>',
    ];

    $form['styles']['new']['operations'] = [
      'data' => [],
    ];
    //  End of styles subform

    //  Start of controls subform
    // Build the list of existing controls for this map.
    $form['controls'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Controls'),
        $this->t('Operations'),
      ],

      '#attributes' => [
        'id' => 'ol-map-controls',
      ],
      '#empty' => t('There are currently no controls defined in this map. Add one by selecting an option below.'),
      '#weight' => 5,
    ];

    // Build the list of new styles that can be added to the map, excluding those that are already attached to the map.
    $new_control_options = [];
    $controls = $this->controlPluginManager->getDefinitions();          //  TODO - generalise !!

    uasort($controls, function ($a, $b) {
      return Unicode::strcasecmp($a['label'], $b['label']);
    });
    foreach ($controls as $control => $definition) {
      $new_control_options[$control] = $definition['label'];
    }


    $map_controls = $this->entity->getControls();

    foreach ($map_controls as $key => $control) {

      if (isset($new_control_options[$control->getPluginId()])) {
        unset($new_control_options[$control->getPluginId()]);  
      }
      
      $form['controls'][$key]['control'] = [
        '#tree' => FALSE,
        'data' => [
          'label' => [
            '#plain_text' => $control->label(),
          ],
        ],
      ];

      $links = [];
      $is_configurable = $control instanceof OpenlayersConfigurablePluginInterface;

      if ($is_configurable) {
        $links['edit'] = [
          'title' => $this->t('Edit'),
          'url' => Url::fromRoute('openlayers.plugin_edit_form', [
            'map' => $this->entity->id(),
            'plugin_type' => 'control',
            'plugin' => $key,
          ]),
        ];
      }
      $links['delete'] = [
        'title' => $this->t('Delete'),
        'url' => Url::fromRoute('openlayers.map.control_delete', [
            'map' => $this->entity->id(),
            'control' => $key,
        ]),
      ];
      $form['controls'][$key]['operations'] = [
        '#type' => 'operations',
        '#links' => $links,
      ];
    }

    // Build the new control addition form and add it to the control list.
    $form['controls']['new'] = [
      '#tree' => FALSE,
    ];
    
    $form['controls']['new']['control'] = [
      'data' => [
        'new_control' => [
          '#type' => 'select',
          '#title' => $this->t('Control'),
          '#title_display' => 'invisible',
          '#options' => $new_control_options,
          '#empty_option' => $this->t('Select a new control'),
        ],
        [
          'add' => [
            '#type' => 'submit',
            '#value' => $this->t('Add control'),
            '#name' => 'add_control',
            '#validate' => ['::controlValidate'],
            '#submit' => ['::submitForm', '::controlSave'],
          ],
        ],
      ],
      '#prefix' => '<div class="map-control-new">',
      '#suffix' => '</div>',
    ];

    $form['controls']['new']['operations'] = [
      'data' => [],
    ];
    //  End of controls subform

    //  Start of interactions subform
    // Build the list of existing interactions for this map.
    $form['interactions'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Interactions'),
        $this->t('Operations'),
      ],

      '#attributes' => [
        'id' => 'ol-map-interactions',
      ],
      '#empty' => t('There are currently no interactions defined in this map. Add one by selecting an option below.'),
      '#weight' => 5,
    ];


    // Build the list of new styles that can be added to the map, excluding those that are already attached to the map.
    $new_interaction_options = [];
    $interactions = $this->interactionPluginManager->getDefinitions();          //  TODO - generalise !!

    uasort($interactions, function ($a, $b) {
      return Unicode::strcasecmp($a['label'], $b['label']);
    });
    foreach ($interactions as $interaction => $definition) {
      $new_interaction_options[$interaction] = $definition['label'];
    }
 

    $map_interactions = $this->entity->getInteractions();

    foreach ($map_interactions as $key => $interaction) {

      if (isset($new_interaction_options[$interaction->getPluginId()])) {
        unset($new_interaction_options[$interaction->getPluginId()]);  
      }
      
      $form['interactions'][$key]['interaction'] = [
        '#tree' => FALSE,
        'data' => [
          'label' => [
            '#plain_text' => $interaction->label(),
          ],
        ],
      ];

      $links = [];
      $is_configurable = $interaction instanceof OpenlayersConfigurablePluginInterface;

      if ($is_configurable) {
        $links['edit'] = [
          'title' => $this->t('Edit'),
          'url' => Url::fromRoute('openlayers.plugin_edit_form', [
            'map' => $this->entity->id(),
            'plugin_type' => 'interaction',
            'plugin' => $key,
          ]),
        ];
      }
      $links['delete'] = [
        'title' => $this->t('Delete'),
        'url' => Url::fromRoute('openlayers.map.interaction_delete', [
            'map' => $this->entity->id(),
            'interaction' => $key,
        ]),
      ];
      $form['interactions'][$key]['operations'] = [
        '#type' => 'operations',
        '#links' => $links,
      ];
    }

    // Build the new interaction addition form and add it to the interaction list.
    $form['interactions']['new'] = [
      '#tree' => FALSE,
    ];
    
    $form['interactions']['new']['interaction'] = [
      'data' => [
        'new_interaction' => [
          '#type' => 'select',
          '#title' => $this->t('Interaction'),
          '#title_display' => 'invisible',
          '#options' => $new_interaction_options,
          '#empty_option' => $this->t('Select a new interaction'),
        ],
        [
          'add' => [
            '#type' => 'submit',
            '#value' => $this->t('Add interaction'),
            '#name' => 'add_interaction',
            '#validate' => ['::interactionValidate'],
            '#submit' => ['::submitForm', '::interactionSave'],
          ],
        ],
      ],
      '#prefix' => '<div class="map-interaction-new">',
      '#suffix' => '</div>',
    ];

    $form['interactions']['new']['operations'] = [
      'data' => [],
    ];
    //  End of interactions subform

    //  Finally set the options for the Openlayers Map View.
    $form['map_view'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Map View'),    
      '#weight' => 50,
    ];

    $form['map_view']['center'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Center'),
    ];
    
    $form['map_view']['center']['lat'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Latitude'),
      '#size' => 20,
    ];

    $form['map_view']['center']['lon'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Longitude'),
      '#size' => 20,
      '#description' => $this->t('Latitude / longitude should be expressed in degrees.'),
    ];

    $zoom_options = ['1' => 1, '2' => 2, '3' => 3];
    
    $form['map_view']['zoom'] = [
      '#type' => 'select',
      '#title' => $this->t('Zoom'),
      '#options' => $zoom_options,
      '#empty_option' => $this->t('Select an initial zoom'),
      '#default_value' => '',
    ];

    $projection_options = [];   //  TODO

    $form['map_view']['projection'] = [
      '#type' => 'select',
      '#title' => $this->t('Projection'),
      '#options' => $projection_options,
      '#empty_option' => $this->t('Select a projection'),
      '#default_value' => '',

    ];

    $form['map_view']['force'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Force'),
      '#description' => $this->t('Forces the initial center / zoom, rather than let the map calculate this automatically based on the features.'),
      '#default_value' => '',
    ];
    
    // Return the form.
    return $form;
  }

  /**
   * Checks for an existing openlayers_map.
   *
   * @param string|int $entity_id
   *   The entity ID.
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return bool
   *   TRUE if this format already exists, FALSE otherwise.
   */
  public function exists($entity_id, array $element, FormStateInterface $form_state) {
    // Use the query factory to build a new openlayers_map entity query.
    $query = $this->entityStorage->getQuery();

    // Query the entity ID to see if its in use.
    $result = $query->condition('id', $element['#field_prefix'] . $entity_id)
      ->execute();

    // We don't need to return the ID, only if it exists or not.
    return (bool) $result;
  }

  /**
   * Overrides Drupal\Core\Entity\EntityFormController::actions().
   *
   * To set the submit button text, we need to override actions().
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   An associative array containing the current state of the form.
   *
   * @return array
   *   An array of supported actions for the current entity form.
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    // Get the basic actins from the base class.
    $actions = parent::actions($form, $form_state);

    // Change the submit button text.
    $actions['submit']['#value'] = $this->t('Save');

    // Return the result.
    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Add code here to validate your config entity's form elements.
    // Nothing to do here.
  }

  /**
   * Overrides Drupal\Core\Entity\EntityFormController::save().
   *
   * Saves the entity. This is called after submit() has built the entity from
   * the form values. Do not override submit() as save() is the preferred
   * method for entity form controllers.
   *
   * @param array $form
   *   An associative array containing the structure of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   An associative array containing the current state of the form.
   */
  public function save(array $form, FormStateInterface $form_state) {

    // EntityForm provides us with the entity we're working on.
    $openlayers_map = $this->getEntity();

    // Drupal already populated the form values in the entity object. Each
    // form field was saved as a public variable in the entity class. PHP
    // allows Drupal to do this even if the method is not defined ahead of
    // time.

    $status = $openlayers_map->save();

    // Grab the URL of the new entity. We'll use it in the message.
    $url = $openlayers_map->toUrl();

    // Create an edit link.
    $edit_link = Link::fromTextAndUrl($this->t('Edit'), $url)->toString();

    if ($status == SAVED_UPDATED) {
      // If we edited an existing entity...
      $this->messenger()->addMessage($this->t('Map %label has been updated.', ['%label' => $openlayers_map->label()]));
      $this->logger('contact')->notice('Map %label has been updated.', ['%label' => $openlayers_map->label(), 'link' => $edit_link]);
    }
    else {
      // If we created a new entity...
      $this->messenger()->addMessage($this->t('Map %label has been added.', ['%label' => $openlayers_map->label()]));
      $this->logger('contact')->notice('Map %label has been added.', ['%label' => $openlayers_map->label(), 'link' => $edit_link]);
    }

    // Redirect the user back to the listing route after the save operation
    $form_state->setRedirect('entity.openlayers_map.list');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Update image effect weights.
    if (!$form_state->isValueEmpty('layers')) {
      $this->updateLayerWeights($form_state->getValue('layers'));
    }
  }

  /**
   * Validate handler for image effect.
   */
  public function layerValidate($form, FormStateInterface $form_state) {
    if (!$form_state->getValue('new_layer')) {
      $form_state->setErrorByName('new_layer', $this->t('Select a layer to add.'));
    }
  }

  /**
   * Submit handler for new/amended layer being added to the map edit form.
   */
  public function layerSave($form, FormStateInterface $form_state) {
    $this->save($form, $form_state);

    // Check if this field has any configuration options.
    //  TODO: potentially convert the layer config options to a plugin (see Image module in core with Styles/Effects as an example 
    $layer = \Drupal::config('openlayers.layer.' . $form_state->getValue('new_layer'));
    
    // Load the configuration form for this option.
    if ($layer->get('is_configurable')) {
      $form_state->setRedirect(
        'openlayers.map.layer_add_form',
        [
          'map' => $this->entity->id(),
          'layer' => $form_state->getValue('new_layer'),
        ],
        ['query' => ['weight' => $form_state->getValue('weight')]]
      );
    }  
    // If there are no configuration options/form, add the layer to the map immediately.
    else {
      $form_state->setRedirect(
        'entity.openlayers_map.edit_form',
        [
          'openlayers_map' => $this->entity->id(),
        ],
      );

      $layer_id = $this->entity->addMapLayer($layer->get());
      $this->entity->save();
      if (!empty($layer_id)) {
        $this->messenger()->addStatus($this->t('The layer was successfully added to the map.'));
      }
    }
  }

  /**
   * Validate handler for map style.
   */
  public function styleValidate($form, FormStateInterface $form_state) {
    if (!$form_state->getValue('new_style')) {
      $form_state->setErrorByName('new_style', $this->t('Select a style to add.'));
    }
  }

  /**
   * Submit handler for new/amended style being added to the map edit form.
   */
  public function styleSave($form, FormStateInterface $form_state) {
    $this->save($form, $form_state);

    // Check if this field has any configuration options.
    $style = $this->stylePluginManager->getDefinition($form_state->getValue('new_style'));


    // Load the configuration form for this option.

    if (is_subclass_of($style['class'], '\Drupal\openlayers\OpenlayersConfigurablePluginInterface')) {

      $form_state->setRedirect(
        'openlayers.style.plugin_add_form',
        [
          'map' => $this->entity->id(),
          'plugin_type' => 'style',
          'plugin' => $form_state->getValue('new_style'),
        ],

      );
    }  
    // If there are no configuration options/form, add the style to the map immediately.
    else {

      $style = [
        'id' => $style['id'],
        'data' => [],
        'weight' => 0,
      ];

      $style_id = $this->entity->addMapStyle($style);

      $this->entity->save();
      if (!empty($style_id)) {
        $this->messenger()->addStatus($this->t('The style was successfully added to the map.'));
      }

      $form_state->setRedirect(
        'entity.openlayers_map.edit_form',
        [
          'openlayers_map' => $this->entity->id(),
        ],
      );
    }
  }

  /**
   * Validate handler for new/amended control.
   */
  public function controlValidate($form, FormStateInterface $form_state) {
    if (!$form_state->getValue('new_control')) {
      $form_state->setErrorByName('new_control', $this->t('Select a control to add.'));
    }
  }

  /**
   * Submit handler for new/amended layer being added to the map edit form.
   */
  public function controlSave($form, FormStateInterface $form_state) {

    $this->save($form, $form_state);

    // Check if this field has any configuration options.
    $control = $this->controlPluginManager->getDefinition($form_state->getValue('new_control'));


    // Load the configuration form for this option.
    if (is_subclass_of($control['class'], '\Drupal\openlayers\OpenlayersConfigurablePluginInterface')) {
      $form_state->setRedirect(
        'openlayers.control.plugin_add_form',
        [
          'map' => $this->entity->id(),
          'plugin_type' => 'control',
          'plugin' => $form_state->getValue('new_control'),
        ],
      );
    }  
    // If there are no configuration options/form, add the control to the map immediately.
    else {
      $control = [
        'id' => $control['id'],
        'data' => [],
        'weight' => 0,
      ];

      $control_id = $this->entity->addMapControl($control);

      $this->entity->save();
      if (!empty($control_id)) {
        $this->messenger()->addStatus($this->t('The control was successfully added to the map.'));
      }

      $form_state->setRedirect(
        'entity.openlayers_map.edit_form',
        [
          'openlayers_map' => $this->entity->id(),
        ],
      );
    }
  }

  /**
   * Validate handler for image effect.
   */
  public function interactionValidate($form, FormStateInterface $form_state) {
    if (!$form_state->getValue('new_interaction')) {
      $form_state->setErrorByName('new_interaction', $this->t('Select a interaction to add.'));
    }
  }

  /**
   * Submit handler for new/amended interaction being added to the map edit form.
   */
  public function interactionSave($form, FormStateInterface $form_state) {

    $this->save($form, $form_state);

    // Check if this field has any configuration options.
    $interaction = $this->interactionPluginManager->getDefinition($form_state->getValue('new_interaction'));


    // Load the configuration form for this option.

    if (is_subclass_of($interaction['class'], '\Drupal\openlayers\OpenlayersConfigurablePluginInterface')) {

      $form_state->setRedirect(
        'openlayers.interaction.plugin_add_form',
        [
          'map' => $this->entity->id(),
          'plugin_type' => 'interaction',
          'plugin' => $form_state->getValue('new_interaction'),
        ],
      );
    }  
    // If there are no configuration options/form, add the interaction to the map immediately.
    else {

      $interaction = [
        'id' => $interaction['id'],
        'data' => [],
        'weight' => 0,
      ];

      $interaction_id = $this->entity->addMapInteraction($interaction);

      $this->entity->save();
      if (!empty($interaction_id)) {
        $this->messenger()->addStatus($this->t('The interaction was successfully added to the map.'));
      }

      $form_state->setRedirect(
        'entity.openlayers_map.edit_form',
        [
          'openlayers_map' => $this->entity->id(),
        ],
      );
    }
  }

  /**
   * Updates layer weights on map.
   *
   * @param array $layers
   *   Associative array with layers having layer uuid as keys and array
   *   with effect data as values.
   */
  protected function updateLayerWeights(array $layers) {
    foreach ($layers as $uuid => $layer_data) {
      if (isset($this->entity->getLayers()[$uuid])) {
        $this->entity->layers[$uuid]['weight'] = $layer_data['weight'];
      }
    }
  }
  
}
