<?php

namespace Drupal\openlayers\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\Core\Cache\Cache;

use Drupal\openlayers\OpenlayersConfigurablePluginInterface;
use Drupal\openlayers\OpenlayersMapInterface;

/**
 * Provides a base form for the settings for Openlayers plugins.
 */
abstract class OpenlayersPluginFormBase extends FormBase {

  /**
   * The Openlayers map.
   *
   * @var \Drupal\image\OpenlayersMapInterface
   */
  protected $olMap;

  /**
   * The openlayers plugin.
   *
   * @var \Drupal\openlayers\OpenlayersPluginInterface.
   */
  protected $olPlugin;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'openlayers_plugin_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param \Drupal\image\ImageStyleInterface $image_style
   *   The openlayers map.
   * @param string $image_effect
   *   The openlayers plugin ID.
   *
   * @return array
   *   The form structure.
   *
   * @throws \Symfony\Component\HttpKernel\Exception\NotFoundHttpException
   */
  public function buildForm(array $form, FormStateInterface $form_state, OpenlayersMapInterface $map = NULL, $plugin_type = NULL, $plugin = NULL) {

    $this->olMap = $map;
    try {
      $this->olPlugin = $this->preparePlugin($plugin_type, $plugin);
    }
    catch (PluginNotFoundException $e) {
      throw new NotFoundHttpException("Invalid plugin id: '$plugin'.");
    }

    $definition = $this->olPlugin->getPluginDefinition();

    $request = $this->getRequest();

    if (!($this->olPlugin instanceof OpenlayersConfigurablePluginInterface)) {
      throw new NotFoundHttpException();
    }

    $form['#attached']['library'][] = 'image/admin';                //  TODO
    $form['uuid'] = [
      '#type' => 'value',
      '#value' => $this->olPlugin->getUuid(),
    ];
    $form['id'] = [
      '#type' => 'value',
      '#value' => $this->olPlugin->getPluginId(),
    ];

    $form['data'] = [];
    $subform_state = SubformState::createForSubform($form['data'], $form, $form_state);
    $form['data'] = $this->olPlugin->buildConfigurationForm($form['data'], $subform_state);
    $form['data']['#tree'] = TRUE;

    // Check the URL for a weight, then the image effect, otherwise use default.
    $form['weight'] = [
      '#type' => 'hidden',
      '#value' => isset($this->olPlugin->ol_id) ? $this->olPlugin->ol_id : $this->olPlugin->getWeight(),
    ];

    // Check the URL for a weight, then the image effect, otherwise use default.
    $form['ol_id'] = [
      '#type' => 'hidden',
      '#value' => isset($definition['ol_id']) ? $definition['ol_id'] : '???',
    ];
    
    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#button_type' => 'primary',
    ];
    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Cancel'),
      '#url' => $this->olMap->toUrl('edit-form'),
      '#attributes' => ['class' => ['button']],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    // The image effect configuration is stored in the 'data' key in the form,
    // pass that through for validation.
    $this->olPlugin->validateConfigurationForm($form['data'], SubformState::createForSubform($form['data'], $form, $form_state));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {

    $form_state->cleanValues();

    // The plugin configuration is stored in the 'data' key in the form, which is then passed through for submission.
    $this->olPlugin->submitConfigurationForm($form['data'], SubformState::createForSubform($form['data'], $form, $form_state));
    $this->olPlugin->setWeight($form_state->getValue('weight'));
    
    if (!$this->olPlugin->getUuid()) {

      $definition = $this->olPlugin->getPluginDefinition();

      $configuration = $this->olPlugin->getConfiguration();

      $configuration['ol_id'] = $form_state->getValue('ol_id');

      $plugin_type = $definition['type'];
      $this->olMap->{'addMap' . ucwords($plugin_type)}($configuration);
    }

    $this->olMap->save();
    //  $node->getCacheTags()

    Cache::invalidateTags(['config:openlayers.map.ol_map2']);
    
    $this->messenger()->addStatus($this->t('The Settings for the plugin were successfully saved.'));
    $form_state->setRedirectUrl($this->olMap->toUrl('edit-form'));
  }

  /**
   * Converts an image effect ID into an object.
   *
   * @param string $image_effect
   *   The image effect ID.
   *
   * @return \Drupal\image\ImageEffectInterface
   *   The image effect object.
   */
  abstract protected function preparePlugin($plugin_type, $plugin);

}
