<?php

namespace Drupal\openlayers\Plugin\Field\FieldFormatter;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

use Drupal\openlayers\OpenlayersService;

/**
 * Base class for the field formatter when displaying an Openlayers map.
 */
abstract class OpenlayersFormatterBase extends FormatterBase {

  /**
   * Openlayers service.
   *
   * @var \Drupal\Openlayers\OpenlayersService
   */
  protected $openlayersService;

  /**
   * OpenlayersFormatterBase constructor.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings settings.
   * @param \Drupal\Openlayers\OpenlayersService $openlayers_service
   *   The Openlayers service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    OpenlayersService $openlayers_service
  ) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);

    $this->openlayersService = $openlayers_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('openlayers.service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'map' => '',
      'height' => 400,
      'height_unit' => 'px',
      'multiple_map' => FALSE,
      'lat' => 0,
      'lon' => 0,
      'zoom' => 10,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $settings = $this->getSettings();

    $form['#tree'] = TRUE;

    // Get the Cardinality set for the Formatter Field.
    $field_cardinality = $this->fieldDefinition->getFieldStorageDefinition()
      ->getCardinality();

    $elements = parent::settingsForm($form, $form_state);
    $field_name = $this->fieldDefinition->getName();

    $mapNames = \Drupal::service('config.storage')->listAll('openlayers.map');
    $map_options = [];
    foreach ($mapNames as $mapName) {
      $mapLabel = \Drupal::config($mapName)->get('label');
      $mapName = str_replace('openlayers.map.', '', $mapName); 
      $map_options[$mapName] = $mapLabel;
    }

    $elements['map'] = [
      '#title' => $this->t('Openlayers Map'),
      '#type' => 'select',
      '#description' => $this->t('Select the map you wish to use.'),
      '#options' => $map_options,
      '#empty_option' => $this->t('-- select map --'),
      '#default_value' => $settings['map'],
      '#required' => TRUE,
      '#weight' => -5,
    ];

    $elements['height'] = [
      '#title' => $this->t('Map Height'),
      '#type' => 'number',
      '#default_value' => $settings['height'],
      '#description' => $this->t('Note: This can be left empty to make the Map fill its parent container height.'),
      '#weight' => -2,
    ];

    $elements['height_unit'] = [
      '#title' => t('Map height unit'),
      '#type' => 'select',
      '#options' => [
        'px' => t('px'),
        '%' => t('%'),
      ],
      '#default_value' => $settings['height_unit'],
      '#description' => t("Whether height is absolute (pixels) or relative (percent).<br><strong>Note:</strong> In case of Percent the Openlayers Map should be wrapped in a container element with defined Height, otherwise won't show up."),
      '#weight' => -1,
    ];   

    $field_cardinality = 1;
    if ($field_cardinality !== 1) {
      $elements['multiple_map'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Multiple Maps'),
        '#description' => $this->t('Check this option if you want to render a separate map for each value in the multi-value field.'),
        '#default_value' => isset($settings['multiple_map']) ? $settings['multiple_map'] : 0,
        '#return_value' => 1,
      ];
    }
    else {
      $elements['multiple_map'] = [
        '#type' => 'hidden',
        '#value' => 0,
      ];
    }
/*    
    $elements['initial_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Initial Settings (for empty map)'),
    ];
*/
//    $elements['initial_settings']['lat'] = [
    $elements['lat'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Latitude'),
//      '#default_value' => isset($settings['initial_settings']['lat']) ? $settings['initial_settings']['lat'] : 0,
      '#default_value' => $settings['lat'],
      '#description' => $this->t('Latitude to be used for a blank map'),
      '#required' => TRUE,
    ];
//    $elements['initial_settings']['lon'] = [
    $elements['lon'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Longtitude'),
//      '#default_value' => isset($settings['initial_settings']['lon']) ? $settings['initial_settings']['lon'] : 0,
      '#default_value' => $settings['lon'],
      '#description' => $this->t('Longitude to be used for a blank map'),
      '#required' => TRUE,
    ];
//    $elements['initial_settings']['zoom'] = [
    $elements['zoom'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default zoom level'),
//      '#default_value' => isset($settings['initial_settings']['zoom']) ? $settings['initial_settings']['zoom'] : 10,
      '#default_value' => $settings['zoom'],
      '#required' => TRUE,
      '#description' => $this->t('Zoom to be used for a single marker or a blank map'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {

    $summary = [];
    $map_label = \Drupal::config('openlayers.map.' . $this->getSetting('map'))->get('label');
    $summary[] = $this->t('Openlayers map: @map', ['@map' => $map_label]);
    $summary[] = $this->t('Map height: @height @height_unit', ['@height' => $this->getSetting('height'), '@height_unit' => $this->getSetting('height_unit')]);

    return $summary;
  }
  
}
