<?php

namespace Drupal\openlayers\Entity;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityWithPluginCollectionInterface;

use Drupal\openlayers\OpenlayersMapInterface;
use Drupal\openlayers\OpenlayersPluginCollection;

/**
 * Defines the Openlayers Map entity.
 *
 * @ConfigEntityType(
 *   id = "openlayers_map",
 *   label = @Translation("Openlayers Map"),
 *   handlers = {
 *     "list_builder" = "Drupal\openlayers\Controller\OpenlayersMapListBuilder",
 *     "form" = {
 *       "add" = "Drupal\openlayers\Form\OpenlayersMapAddForm",
 *       "edit" = "Drupal\openlayers\Form\OpenlayersMapEditForm",
 *       "delete" = "Drupal\openlayers\Form\OpenlayersMapDeleteForm",
 *     }
 *   },
 *   config_prefix = "map",
 *   admin_permission = "administer openlayers",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "layers",
 *     "styles",
 *     "controls",
 *     "interactions",
 *     "map_view"
 *   },
 *   links = {
 *     "edit-form" = "/admin/config/system/openlayers/map/{openlayers_map}/edit",
 *     "delete-form" = "/admin/config/system/openlayers/map/{openlayers_map}/delete",
 *   }
 * )
 */
class OpenlayersMap extends ConfigEntityBase implements OpenlayersMapInterface, EntityWithPluginCollectionInterface {

  /**
   * The Map ID.
   *
   * @var string
   */
  public $id;

  /**
   * The Map label.
   *
   * @var string
   */
  public $label;

  // Your specific configuration property get/set methods go here,
  // implementing the interface.

  /**
   * The ??? flag.?????
   *
   * @var array ???
   */
  public $layers;
  
  /**
   * The ??? flag.?????
   *
   * @var array ???
   */
  public $styles;
  
/**
   * The ??? flag.?????
   *
   * @var array ???
   */
  public $controls;

  /**
   * The ??? flag.?????
   *
   * @var array ???
   */
  public $interactions;
  
  /**
   * The ??? flag.?????
   *
   * @var array ???
   */
  public $map_view;
  
  /**
   * {@inheritdoc}
   */
  public function id() {
    return $this->id;
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    return $this->label;
  }

  /**
   * Returns the image effect plugin manager.
   *
   * @return \Drupal\Component\Plugin\PluginManagerInterface
   *   The image effect plugin manager.
   */
  protected function getStylePluginManager() {
    return \Drupal::service('plugin.manager.openlayers.style');
  }

  /**
   * Returns the image effect plugin manager.
   *
   * @return \Drupal\Component\Plugin\PluginManagerInterface
   *   The image effect plugin manager.
   */
  protected function getControlPluginManager() {
    return \Drupal::service('plugin.manager.openlayers.control');
  }
  
  /**
   * Returns the image effect plugin manager.
   *
   * @return \Drupal\Component\Plugin\PluginManagerInterface
   *   The image effect plugin manager.
   */
  protected function getInteractionPluginManager() {
    return \Drupal::service('plugin.manager.openlayers.interaction');
  }
  
  /**
   * {@inheritdoc}
   */
  public function getPluginCollections() {
    return ['styles' => $this->getStyles(), 'controls' => $this->getControls(), 'interactions' => $this->getInteractions()];
  }
  
  /**
   * {@inheritdoc}
   */
  public function getLayer($layer) {
//    return $this->getLayers()->get($layer);
    return $this->getLayers()[$layer];
  }

  /**
   * {@inheritdoc}
   */
  public function getLayers() {
    if (!isset($this->layers)) {
      return [];
    }
    
    $layer_collection = [];
    foreach ($this->layers as $key => $layer) {
      $layer_id = $key;
    }
    return $this->layers;
  }

  /**
   * {@inheritdoc}
   */
  public function getAllLayers() {
    $layerNames = \Drupal::service('config.storage')->listAll('openlayers.layer');

    $layers = [];

    foreach ($layerNames as $layerName) {
      $layerLabel = \Drupal::config($layerName)->get('label');
      $layerName = str_replace('openlayers.layer.', '', $layerName); 
      $layers[$layerName] = $layerLabel;
    }

    return $layers;
  }

  /**
   * {@inheritdoc}
   */
  public function getStyle($style) {
    return $this->getStyles()->get($style);
  }
  
  /**
   * {@inheritdoc}
   */
  public function getStyles() {
    if (!isset($this->stylesCollection)) {
      $this->stylesCollection = new OpenlayersPluginCollection($this->getStylePluginManager(), (isset($this->styles) ? $this->styles : []));
      $this->stylesCollection->sort();
    }
    return $this->stylesCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function getControl($control) {
    return $this->getControls()->get($control);
  }
  
  /**
   * {@inheritdoc}
   */
  public function getControls() {
    if (!isset($this->controlsCollection)) {
      $this->controlsCollection = new OpenlayersPluginCollection($this->getControlPluginManager(), (isset($this->controls) ? $this->controls : []));
      $this->controlsCollection->sort();
    }
    return $this->controlsCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function getInteraction($interaction) {
    return $this->getInteractions()->get($interaction);
  }
  
  /**
   * {@inheritdoc}
   */
  public function getInteractions() {
    if (!isset($this->interactionsCollection)) {
      $this->interactionsCollection = new OpenlayersPluginCollection($this->getInteractionPluginManager(), (isset($this->interactions) ? $this->interactions : []));
      $this->interactionsCollection->sort();
    }
    return $this->interactionsCollection;
  }

  /**
   * {@inheritdoc}
   */
  public function addMapLayer($configuration) {
    $uuid = \Drupal::service('uuid')->generate();
    $this->layers[$configuration['uuid']] = $configuration;
    return $uuid;
  }

  /**
   * {@inheritdoc}
   */
  public function addMapStyle($configuration) {
    $configuration['uuid'] = $this->uuidGenerator()->generate();
    $this->getStyles()->addInstanceId($configuration['uuid'], $configuration);
    return $configuration['uuid'];
  }

  /**
   * {@inheritdoc}
   */
  public function addMapControl($configuration) {
    $configuration['uuid'] = $this->uuidGenerator()->generate();
    $this->getControls()->addInstanceId($configuration['uuid'], $configuration);
    return $configuration['uuid'];
  }

  /**
   * {@inheritdoc}
   */
  public function addMapInteraction($configuration) {
    $configuration['uuid'] = $this->uuidGenerator()->generate();
    $this->getInteractions()->addInstanceId($configuration['uuid'], $configuration);
    return $configuration['uuid'];
  }
  
  /**
   * {@inheritdoc}
   */
  public function updateMapLayer($configuration) {
    $this->layers[$configuration['uuid']] = $configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMapLayer($configuration) {
    $uuid = $configuration->uuid;
    unset($this->layers[$uuid]);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMapStyle($configuration) {
    $uuid = $configuration->uuid;
    unset($this->styles[$uuid]);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMapControl($configuration) {
    $uuid = $configuration->uuid;
    unset($this->controls[$uuid]);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMapInteraction($configuration) {
    $uuid = $configuration->uuid;
    unset($this->interactions[$uuid]);
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    parent::postSave($storage, $update);
    if ($update) {
      if (!empty($this->original) && $this->id() !== $this->original->id()) {
        // The old image style name needs flushing after a rename.
        $this->original->flush();
      }
      else {
        // Flush image style when updating without changing the name.
        $this->flush();
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function flush() {
    // Clear caches so that formatters may be added for this style.
    Cache::invalidateTags($this->getCacheTagsToInvalidate());

    return $this;
  }
  
}
