<?php

namespace Drupal\openlayers\Plugin\Field\FieldWidget;

use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

use Drupal\openlayers\OpenlayersService;
use Drupal\openlayers\MapSettings;

/**
 * Base class for the field widget when displaying an Openlayers map.
 */
abstract class OpenlayersWidgetBase extends WidgetBase {
  
  /**
   * The geoPhpWrapper service.
   *
   * @var \Drupal\openlayers\OpenlayersService
   */
  protected $openlayersService;

  /**
   * OpenlayersWidget constructor.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param array $third_party_settings
   *   Any third party settings settings.
   * @param \Drupal\openlayers\OpenlayersService $openlayers_service
   *   The Openlayers service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    OpenlayersService $openlayers_service
  ) {
    parent::__construct(
      $plugin_id,
      $plugin_definition,
      $field_definition,
      $settings,
      $third_party_settings
    );
    $this->openlayersService = $openlayers_service;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('openlayers.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'map' => '',
      'height' => 400,
      'height_unit' => 'px',
      'multiple_map' => FALSE,
      'initial_settings' => [
        'lat' => 0.0,
        'lon' => 0.0,
        'zoom' => 10,
      ],
      'input' => [
        'show' => FALSE,
        'editable' => FALSE,
      ],
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $settings = $this->getSettings();
    $form['#tree'] = TRUE;  
    $field_name = $this->fieldDefinition->getName();

    $mapNames = \Drupal::service('config.storage')->listAll('openlayers.map');
    $map_options = [];
    foreach ($mapNames as $mapName) {
      $mapLabel = \Drupal::config($mapName)->get('label');
      $mapName = str_replace('openlayers.map.', '', $mapName); 
      $map_options[$mapName] = $mapLabel;
    }

    $form['map'] = [
      '#title' => $this->t('Openlayers Map'),
      '#type' => 'select',
      '#description' => $this->t('Select the map you wish to use.'),
      '#options' => $map_options,
      '#empty_option' => $this->t('-- select map --'),
      '#default_value' => $settings['map'],
      '#required' => TRUE,
      '#weight' => -5,
    ];

    $form['height'] = [
      '#title' => $this->t('Map Height'),
      '#type' => 'number',
      '#default_value' => $settings['height'],
      '#description' => $this->t('Note: This can be left empty to make the Map fill its parent container height.'),
      '#weight' => -2,
    ];

    $form['height_unit'] = [
      '#title' => t('Map height unit'),
      '#type' => 'select',
      '#options' => [
        'px' => t('px'),
        '%' => t('%'),
      ],
      '#default_value' => $settings['height_unit'],
      '#description' => t("Whether height is absolute (pixels) or relative (percent).<br><strong>Note:</strong> In case of Percent the Openlayers Map should be wrapped in a container element with defined Height, otherwise won't show up."),
      '#weight' => -1,
    ];    

    $form['initial_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Initial Settings (for empty map)'),
    ];

    $form['initial_settings']['lat'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Latitude'),
      '#default_value' => isset($settings['initial_settings']['lat']) ? $settings['initial_settings']['lat'] : 0,
      '#required' => TRUE,
    ];
    $form['initial_settings']['lon'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Longtitude'),
      '#default_value' => isset($settings['initial_settings']['lon']) ? $settings['initial_settings']['lon'] : 0,
      '#required' => TRUE,
    ];
    $form['initial_settings']['zoom'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Default zoom level'),
      '#default_value' => isset($settings['initial_settings']['zoom']) ? $settings['initial_settings']['zoom'] : 10,
      '#required' => TRUE,
      '#description' => $this->t('Zoom to be used for a single marker or a blank map'),
    ];

    $form['input'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Geofield Settings'),
    ];
    $form['input']['show'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show geofield input element'),
      '#default_value' => $settings['input']['show'],
    ];
    $form['input']['editable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow editing of geofield input field'),
      '#default_value' => $settings['input']['editable'],
      '#states' => [
        'invisible' => [
          ':input[name="fields[field_geofield][settings_edit_form][settings][input][show]"]' => ['checked' => FALSE],   //  TODO - fieldname ?
        ],
      ],
    ];

    $form['toolbar'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Editing Toolbar settings'),
    ];

    $form['toolbar']['position'] = [
      '#type' => 'select',
      '#title' => $this->t('Toolbar position.'),
      '#options' => [
        'top' => 'top',
        'bottom' => 'bottom',
        'left' => 'left',
        'right' => 'right',
        'topleft' => 'topleft',
        'topright' => 'topright',
        'bottomleft' => 'bottomleft',
        'bottomright' => 'bottomright',
      ],
      '#default_value' => $settings['toolbar']['position'],
    ];

    return $form;
  }

}
