<?php

namespace Drupal\timefield\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Template\Attribute;

/**
 * Plugin for the 'timefield_duration_formatter' field formatter.
 *
 * @FieldFormatter(
 *  id = "timefield_duration_formatter",
 *  label = @Translation("Duration"),
 *  field_types = {"timefield"}
 * )
 */
class TimeFieldDuration extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {

    return [
      'duration_format' => 'minutes',
      'display_format' => [
        'hour' => 'g',
        'minute' => 'i',
        'separator' => ':',
        'period' => 'a',
        'periodSeparator' => '',
      ],
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    // @todo Change the autogenerated stub
    $element = parent::settingsForm($form, $form_state);
    $settings = $this->getSettings();
    $element['duration_format'] = [
      '#title' => $this->t('Time Duration Format'),
      '#type' => 'select',
      '#options' => _timefield_duration_options(),
      '#default_value' => $settings['duration_format'],
      '#required' => TRUE,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    // @todo Change the autogenerated stub
    $summary = parent::settingsSummary();
    $settings = $this->getSettings();
    $field_settings = $this->getFieldSettings();

    if (empty($field_settings['totime'])) {
      $summary[] = $this->t('This display format is invalid for the current field configuration');
    }
    else {
      $summary[] = $this->t('Current Format: @format', [
        '@format' => _timefield_duration_options($settings['duration_format']),
      ]);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $settings = $this->getSettings();
    $field_settings = $this->getFieldSettings();
    $label_hidden = ($this->label == 'hidden');
    foreach ($items as $delta => $item) {
      $item = $items->get($delta);
      if (!empty($item->value)) {
        $value = $item->getValue('value');
        $value['value'] = trim(timefield_integer_to_time($settings['display_format'], $item->value));
        if (isset($item->value2)) {
          $value['value2'] = $item->value2;
          $value['formatted_value2'] = trim(timefield_integer_to_time($settings['display_format'], $item->value2));
          $value['duration'] = timefield_time_to_duration($item->value, $item->value2, $settings['duration_format']);
          $value['time'] = timefield_time_to_duration($item->value, $item->value2, $settings['duration_format']);
        }
        else {
          $value['time'] = 0;
        }
        $item->setValue($value);
        $items->set($delta, $value);
      }
    }
    $element = [
      '#theme' => 'timefield_duration',
      '#label' => $this->fieldDefinition->get('label'),
      '#label_display' => $this->label,
      '#label_hidden' => $label_hidden,
      '#items' => $items,
      '#settings' => $field_settings,
      '#format' => 'duration',
      '#attributes' => new Attribute(
          [
            'class' => 'timefield',
          ]
      ),
    ];
    return $element;
  }

}
